#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# 26-09-17

# Name of job and destinations for outputs

#SBATCH --job-name=resse48B
#SBATCH --output=StdOut.o.%j
#SBATCH --error=StdErr.e.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=rc-ph1117
#SBATCH --time=24:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

# change to the scratch directory and make subdirectory for the results
# back up this input file in the results directory
 
echo Running on host `hostname` 
echo Start time is `date`
echo Directory is `pwd`

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0

export OMP_NUM_THREADS=1

TMP_DIR='/home/e/pysdw/scratch/tmp_ph_48B'
PSEUDO_DIR='/home/e/pysdw/pseudo'
RESULTS_DIR='/home/e/pysdw/resse/results_ph_48B'

# not a restart

mkdir $TMP_DIR
mkdir $RESULTS_DIR
cp $0 $RESULTS_DIR
cd $RESULTS_DIR

# How to run QE codes:

PW_COMMAND='mpirun -np 64 pw.x '
PH_COMMAND='mpirun -np 64 ph.x '
DYNMAT_COMMAND='mpirun -np 1 dynmat.x '

# computational parameters (default values commented out for reference)

ecut=60.00       #60
rcut=240.00      #300
elthresh=1.0d-10 #1E-12
fthresh=1.0d-4   # 1E-4
ethresh=1.0d-5   #1E-5
nks=4            # 4
phthresh=1.0d-11 #1.0d-12
pthresh=0.5      #0.5

mass1=186.21     # Re
mass2=78.96      # Se
mass3=32.065     # S

atom1='Re  '$mass1'  Re.pz-n-nc.UPF'
atom2='Se  '$mass2'  Se.pz-n-nc.UPF'
atom3='S   '$mass3'  S.pz-n-nc.UPF'

#
# Initial vc-relax calculation with cell parameters and atomic positions expressed in crystal coordinates
#

cat > ReSSe.scf.in << EOF
 &control
    calculation='scf',
    max_seconds=86000,
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=13.194,
    nat=  48, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut,
    nbnd= 156,
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
ATOMIC_SPECIES
$atom1
$atom2
$atom3
CELL_PARAMETERS (alat= 24.93304660)
   0.999066560   0.001311050  -0.000699560
  -0.490096925   0.892100697   0.000789828
  -0.131315453  -0.091187919   0.478109177
ATOMIC_POSITIONS (crystal)
Re       0.032051528   0.157993801   0.489067604
Re       0.244635142   0.144999933   0.497960099
Se       0.125623203   0.191434141   0.200005527
S        0.363528606   0.171525359   0.261325270
Se       0.112296585   0.053209018   0.724607737
Se       0.378110832   0.067679322   0.701632700
Re       0.530453308   0.158878790   0.492283499
Re       0.742092006   0.144472058   0.494672855
Se       0.621288011   0.191493105   0.199947478
Se       0.858004132   0.167442099   0.232544175
Se       0.612043936   0.053487974   0.724243267
Se       0.877798585   0.068410653   0.695485829
Re       0.031658392   0.658070747   0.490392817
Re       0.242808329   0.644491814   0.495714904
Se       0.123262205   0.691488549   0.199506026
Se       0.357967599   0.667267359   0.233137821
Se       0.112198499   0.553014430   0.724842357
Se       0.378080591   0.568367928   0.697463491
Re       0.531715334   0.658148561   0.490117779
Re       0.742727319   0.644600506   0.495187420
Se       0.623108433   0.691435402   0.198802297
Se       0.858211402   0.667345074   0.233038709
Se       0.612904187   0.554074810   0.725694752
Se       0.877825956   0.568339041   0.697067871
Re      -0.031737514  -0.158174216  -0.492681993
Re      -0.243141863  -0.144632974  -0.498342630
Se      -0.123491538  -0.191437457  -0.202013726
Se      -0.358644569  -0.167323486  -0.236217777
Se      -0.112391797  -0.053475203  -0.727588808
Se      -0.378566083  -0.068435398  -0.699845975
Re       0.467729628  -0.157949708  -0.493199886
Re       0.256960809  -0.144827705  -0.496727855
Se       0.376719125  -0.191761921  -0.201100018
Se       0.141350628  -0.167825394  -0.234832366
Se       0.386663698  -0.052354175  -0.726269715
Se       0.122924193  -0.067705026  -0.698766371
Re      -0.032199468   0.341335284  -0.492439811
Re      -0.243366301   0.355199820  -0.497042738
Se      -0.123711554   0.307967717  -0.201480928
Se      -0.357281827   0.333058153  -0.231961341
Se      -0.112774310   0.446395534  -0.726418777
Se      -0.378781854   0.431046665  -0.697916443
Re       0.468400483   0.341155657  -0.488233703
Re       0.257451139   0.355069216  -0.495831829
Se       0.376439740   0.308119194  -0.197383375
Se       0.141048824   0.331679527  -0.235160189
Se       0.387844411   0.445967512  -0.725022921
Se       0.122161881   0.431237913  -0.698263110
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

#$ECHO " running the scf calculation ; \c"
#$PW_COMMAND -input ReSSe.scf.in &> ReSSe.scf.out
#$ECHO " done"

cat > ReSSe.ph.in <<EOF
phonons of ReSSe
&inputph
   recover=.true.,
   alpha_mix(1)=0.3,
   tr2_ph=$phthresh,
   max_seconds=86000,
   prefix='ReSSe',
   ldisp=.false.,
   trans=.true.,
   epsil=.false.,
   lraman=.false.,
   amass(1)=186.21,
   amass(2)=78.96,
   amass(3)=32.065,
   outdir='$TMP_DIR/',
   fildyn='ReSSe.dyn',
/
0.0   0.0   0.0
EOF

$ECHO " running the phonon calculation"
$PH_COMMAND -input ReSSe.ph.in &> ReSSe.ph.out
$ECHO " done"

cat > dynmat.in <<EOF
&input
   fildyn='ReSSe.dyn',
   q(1)=0.0,
   q(2)=0.0,
   q(3)=0.0,
   asr='simple'
/
EOF

$ECHO " running dynmat"
$DYNMAT_COMMAND -input dynmat.in &> dynmat.out
$ECHO " done"


