#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# 26-09-17

# Name of job and destinations for outputs

#SBATCH --job-name=resse48C
#SBATCH --output=StdOut.o.%j
#SBATCH --error=StdErr.e.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=rc-ph1117
#SBATCH --time=24:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

# change to the scratch directory and make subdirectory for the results
# back up this input file in the results directory
 
echo Running on host `hostname` 
echo Start time is `date`
echo Directory is `pwd`

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0

export OMP_NUM_THREADS=1

TMP_DIR='/home/e/pysdw/scratch/tmp_ph_48C'
PSEUDO_DIR='/home/e/pysdw/pseudo'
RESULTS_DIR='/home/e/pysdw/resse/results_ph_48C'

# not a restart

mkdir $TMP_DIR
mkdir $RESULTS_DIR
cp $0 $RESULTS_DIR
cd $RESULTS_DIR

# How to run QE codes:

PW_COMMAND='mpirun -np 64 pw.x'
PH_COMMAND='mpirun -np 64 ph.x'
DYNMAT_COMMAND='mpirun -np 1 dynmat.x'

# computational parameters (default values commented out for reference)

ecut=60.00       #60
rcut=240.00      #300
elthresh=1.0d-10 #1E-12
fthresh=1.0d-4   # 1E-4
ethresh=1.0d-5   #1E-5
nks=4            # 4
phthresh=1.0d-11 #1.0d-12
pthresh=0.5      #0.5

mass1=186.21     # Re
mass2=78.96      # Se
mass3=32.065     # S

atom1='Re  '$mass1'  Re.pz-n-nc.UPF'
atom2='Se  '$mass2'  Se.pz-n-nc.UPF'
atom3='S   '$mass3'  S.pz-n-nc.UPF'

#
# Initial vc-relax calculation with cell parameters and atomic positions expressed in crystal coordinates
#

cat > ReSSe.scf.in << EOF
 &control
    calculation='scf',
    max_seconds=86000,
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=13.194,
    nat=  48, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut,
    nbnd= 156,
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='local-TF',
    mixing_beta = 0.7,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
ATOMIC_SPECIES
$atom1
$atom2
$atom3
CELL_PARAMETERS (alat= 24.93304660)
   0.999709472   0.000916373  -0.000060672
  -0.490762360   0.891208431  -0.001152763
  -0.131054529  -0.091918717   0.478300233
ATOMIC_POSITIONS (crystal)
Re       0.031750869   0.156617281   0.489418737
Re       0.240883078   0.142939534   0.495416709
Se       0.123373002   0.190466853   0.197931605
Se       0.358064508   0.167037426   0.235235786
S        0.107554821   0.051833982   0.698303166
Se       0.377172891   0.068459890   0.698536403
Re       0.531695995   0.158225910   0.492414967
Re       0.743340429   0.144665032   0.497919107
Se       0.623858480   0.191477003   0.202340882
Se       0.858577435   0.167444411   0.235103155
Se       0.612534641   0.053767276   0.726873707
Se       0.879312294   0.068728562   0.698176538
Re       0.032458714   0.659127702   0.493467019
Re       0.242912159   0.644500320   0.497247657
Se       0.123170045   0.690666542   0.201215301
Se       0.358287036   0.666925439   0.234859789
Se       0.112827367   0.553938888   0.727103866
Se       0.377867763   0.568195597   0.698050869
Re       0.532079124   0.658253485   0.491858325
Re       0.743823092   0.645564467   0.498204721
Se       0.623383710   0.690759820   0.200922390
Se       0.859232153   0.668264307   0.236581373
Se       0.612764193   0.554031662   0.726827919
Se       0.878744015   0.569698080   0.700127948
Re      -0.030505817  -0.155413808  -0.492389456
Re      -0.242643989  -0.144260916  -0.496589742
Se      -0.121401087  -0.188742290  -0.200202788
Se      -0.357875882  -0.166822201  -0.233777748
Se      -0.114742490  -0.054592226  -0.730644910
Se      -0.378139224  -0.068648215  -0.697257855
Re       0.468243895  -0.158294321  -0.490358972
Re       0.257195539  -0.144665560  -0.496102288
Se       0.376708639  -0.191065358  -0.199454298
Se       0.142496455  -0.164626677  -0.234055373
Se       0.387840310  -0.053694582  -0.724848506
Se       0.121881008  -0.069663137  -0.701925966
Re      -0.031790606   0.341788896  -0.489837223
Re      -0.242501240   0.355695795  -0.494083399
Se      -0.122938333   0.308999120  -0.198419785
Se      -0.358293629   0.332838593  -0.232314504
Se      -0.112116186   0.446808877  -0.723283010
Se      -0.377580488   0.431607704  -0.695454768
Re       0.468018659   0.341327202  -0.489624439
Re       0.256657608   0.354496187  -0.496794701
Se       0.374830419   0.307084973  -0.200272855
Se       0.141388990   0.331351834  -0.234558863
Se       0.387837124   0.445996974  -0.724803305
Se       0.121762508   0.430903665  -0.697083183
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

#$ECHO " running the scf calculation ; \c"
#$PW_COMMAND -input ReSSe.scf.in &> ReSSe.scf.out
#$ECHO " done"

cat > ReSSe.ph.in <<EOF
phonons of ReSSe
&inputph
   recover=.true.,
   alpha_mix(1)=0.3,
   tr2_ph=$phthresh,
   max_seconds=86000,
   prefix='ReSSe',
   ldisp=.false.,
   trans=.true.,
   epsil=.false.,
   lraman=.false.,
   amass(1)=186.21,
   amass(2)=78.96,
   amass(3)=32.065,
   outdir='$TMP_DIR/',
   fildyn='ReSSe.dyn',
/
0.0   0.0   0.0
EOF

$ECHO " running the phonon calculation"
$PH_COMMAND -input ReSSe.ph.in &> ReSSe.ph.out
$ECHO " done"

cat > dynmat.in <<EOF
&input
   fildyn='ReSSe.dyn',
   q(1)=0.0,
   q(2)=0.0,
   q(3)=0.0,
   asr='simple'
/
EOF

$ECHO " running dynmat"
$DYNMAT_COMMAND -input dynmat.in &> dynmat.out
$ECHO " done"


