#!/bin/bash
# lines starting with #SBATCH are options for the sbatch command
# 26-09-17

# Name of job and destinations for outputs

#SBATCH --job-name=resse48D
#SBATCH --output=StdOut.o.%j
#SBATCH --error=StdErr.e.%j

# Number of nodes (here 4 nodes with 16 CPUs each)
# The total number of nodes passed to mpirun will be nodes*ppn

#SBATCH --nodes=4
#SBATCH --ntasks-per-node=16

# Specify the account type and usage limits

#SBATCH --account=rc-ph1117
#SBATCH --time=24:00:00

#SBATCH --mail-user=pysdw@bath.ac.uk
#SBATCH --mail-type=ALL

# change to the scratch directory and make subdirectory for the results
# back up this input file in the results directory
 
echo Running on host `hostname` 
echo Start time is `date`
echo Directory is `pwd`

module purge
module load untested
module avail
module load slurm
module load espresso/5.2.0

export OMP_NUM_THREADS=1

TMP_DIR='/home/e/pysdw/scratch/tmp_ph_48D'
PSEUDO_DIR='/home/e/pysdw/pseudo'
RESULTS_DIR='/home/e/pysdw/resse/results_ph_48D'

# a restart

#mkdir $TMP_DIR
#mkdir $RESULTS_DIR
cp $0 $RESULTS_DIR
cd $RESULTS_DIR

# How to run QE codes:

PW_COMMAND='mpirun -np 64 pw.x'
PH_COMMAND='mpirun -np 64 ph.x'
DYNMAT_COMMAND='mpirun -np 1 dynmat.x'

# computational parameters (default values commented out for reference)

ecut=60.00       #60
rcut=240.00      #300
elthresh=1.0d-10 #1E-12
fthresh=1.0d-4   # 1E-4
ethresh=1.0d-5   #1E-5
nks=4            # 4
phthresh=1.0d-11 #1.0d-12
pthresh=0.5      #0.5

mass1=186.21     # Re
mass2=78.96      # Se
mass3=32.065     # S

atom1='Re  '$mass1'  Re.pz-n-nc.UPF'
atom2='Se  '$mass2'  Se.pz-n-nc.UPF'
atom3='S   '$mass3'  S.pz-n-nc.UPF'

#
# Initial vc-relax calculation with cell parameters and atomic positions expressed in crystal coordinates
#

cat > ReSSe.scf.in << EOF
 &control
    calculation='scf',
    max_seconds=86000,
    restart_mode='from_scratch',
    prefix='ReSSe',
    pseudo_dir = '$PSEUDO_DIR',
    outdir='$TMP_DIR',
    forc_conv_thr=$fthresh,
    etot_conv_thr=$ethresh,
    tprnfor=.true.,
    tstress=.true.,
 /
 &system    
    ibrav=  0, 
    A=13.194,
    nat=  48, ntyp= 3, 
    ecutwfc =$ecut,
    ecutrho = $rcut,
    nbnd= 156,
 /
 &electrons
    conv_thr =  $elthresh,
    mixing_mode='plain',
    mixing_beta = 0.3,
 /
 &ions
    ion_dynamics='bfgs',
 /
&cell
    cell_dynamics='bfgs',
 /
ATOMIC_SPECIES
$atom1
$atom2
$atom3
CELL_PARAMETERS (alat= 24.93304660)
   0.999119537   0.001383743  -0.000954119
  -0.490058244   0.890698922  -0.000427866
  -0.131453899  -0.091793314   0.478925856
ATOMIC_POSITIONS (crystal)
Re       0.032323030   0.158200822   0.492536506
Re       0.243882417   0.142931045   0.495121083
Se       0.123852574   0.190952828   0.202483737
Se       0.357973064   0.167864115   0.229918556
Se       0.113652875   0.053438108   0.725838535
S        0.373539426   0.066508036   0.668696542
Re       0.529131596   0.156681090   0.490195086
Re       0.742580921   0.144568253   0.498377351
Se       0.622528284   0.190682104   0.202157715
Se       0.858404169   0.167522893   0.236865069
Se       0.611182958   0.053587446   0.726505722
Se       0.878435562   0.068749841   0.699054405
Re       0.032699302   0.659282287   0.493074709
Re       0.243382957   0.645663288   0.497857251
Se       0.123916601   0.692445801   0.202498234
Se       0.357996188   0.667130092   0.235007460
Se       0.113339330   0.554560672   0.727480492
Se       0.378453717   0.568958171   0.699013333
Re       0.531734742   0.658155445   0.491689545
Re       0.742934344   0.644532889   0.497016845
Se       0.623183963   0.691127364   0.200984830
Se       0.858243323   0.667029352   0.235280555
Se       0.612384991   0.553167551   0.725651367
Se       0.878387856   0.568816364   0.698872366
Re      -0.031795760  -0.158162653  -0.490775599
Re      -0.242853209  -0.144640606  -0.495692830
Se      -0.123095152  -0.191223962  -0.199815856
Se      -0.358186842  -0.167044446  -0.234018496
Se      -0.112700586  -0.053542546  -0.724857783
Se      -0.377881612  -0.068498291  -0.697067549
Re       0.468265921  -0.157840521  -0.491135740
Re       0.258180078  -0.141664789  -0.497799404
Se       0.377160833  -0.189052061  -0.200755348
Se       0.143343137  -0.165017117  -0.234089305
Se       0.387536364  -0.054640437  -0.730000218
Se       0.120111577  -0.069480200  -0.701238995
Re      -0.031418093   0.341979714  -0.488462285
Re      -0.243216480   0.355117491  -0.495327849
Se      -0.123457206   0.308895192  -0.198531307
Se      -0.359296748   0.331483078  -0.234190400
Se      -0.111924538   0.446765452  -0.722832733
Se      -0.378153102   0.431145110  -0.697571093
Re       0.468050302   0.341049465  -0.491454022
Re       0.257676717   0.354812848  -0.494386084
Se       0.377323508   0.306749856  -0.200378806
Se       0.142487111   0.332587810  -0.232361822
Se       0.387207424   0.446089812  -0.724758158
Se       0.122492166   0.431575943  -0.694675612
K_POINTS {automatic}
 $nks $nks $nks 0 0 0
EOF

#$ECHO " running the scf calculation ; \c"
#$PW_COMMAND -input ReSSe.scf.in &> ReSSe.scf.out
#$ECHO " done"

cat > ReSSe.ph.in <<EOF
phonons of ReSSe
&inputph
   recover=.true.,
   alpha_mix(1)=0.3,
   tr2_ph=$phthresh,
   max_seconds=86000,
   prefix='ReSSe',
   ldisp=.false.,
   trans=.true.,
   epsil=.false.,
   lraman=.false.,
   amass(1)=186.21,
   amass(2)=78.96,
   amass(3)=32.065,
   outdir='$TMP_DIR/',
   fildyn='ReSSe.dyn',
/
0.0   0.0   0.0
EOF

$ECHO " running the phonon calculation"
$PH_COMMAND -input ReSSe.ph.in &> ReSSe.ph.out
$ECHO " done"

cat > dynmat.in <<EOF
&input
   fildyn='ReSSe.dyn',
   q(1)=0.0,
   q(2)=0.0,
   q(3)=0.0,
   asr='simple'
/
EOF

$ECHO " running dynmat"
$DYNMAT_COMMAND -input dynmat.in &> dynmat.out
$ECHO " done"


