/*
	Simulates Kawasaki-dynamics with a Kinetic-Monte-Carlo-algorithm and periodic boundary-conditions
*/


#include <iostream>		// needed for output in control-window
#include <string>		// needed for strings/names
#include <fstream>		// needed for output to external files
#include <vector>		// needed for vectors/vector-operations
#include <random>		// needed for random-number-generator
#include <numeric>		// needed for accumulate or innerproduct
#include <math.h>		// needed for pow, exp
#include <iomanip>		// needed for setprecision

using namespace std;

#include "binsearch.h"
#include "modulo.h"		// for finding direction
#include "getneighbour.h"	// for finding neighbour
#include "getlocalmean.h"	// for local mean

void Kawasaki_Ising_KMC( vector<int>* S_adr, vector<double>* rate_adr, int l_rate, double dT, int Nx, int Ny, double tilt, double b, double h, string boundarycondition )
{
	double time = 0;					// initial physical time
	vector<double> cumsum(l_rate);		// cumulated sum of jump-rates of the sites in order of rate (i.e. 1left, 1right, 2left, 2right,...)
	double lastcumsum = 0;				// last element of cumsum (i.e. sum over all jump-rates)

	// for writing of evolution-file:
	char filename[100];
	sprintf(filename, "Spin-Evolution (%dx%d, tilt=%1.2f, T=%1.2f, h=%1.1f, t=%d, time=%1.3f).txt", Ny, Nx, tilt, 1/b, h, 0, time );

	// initialize random distribution:
	random_device rd;
	mt19937_64 seed(rd());
	uniform_real_distribution<double> distribution(0.0,1.0);
	double randno1 = 0, randno2 = 0;	// initializes random number between 0 and 1
	double randno3 = 0;

	// initialize loop-variables:
	int nnndirection = -1, nndirection = -1, direction = -1, backnnndirection = -1;	// initialize variable, that decides if to jump N, S, E or W (0 for N, 3 for W clockwise like in jump-rates-list)
	int spinmemory = 0, lowerseam = (Ny-1)*Nx, ujumpx = -1, ljumpx = -1;
	int rateindex = -1, jumpx = -1, jumptox = -1, nn = -1, nnn = -1;
	double searchat = 0.0, guess = -1.;
	double localmean = 0, nnenergygain, nnnenergygain, localrate;

	while( time <= dT )								// run until physical time "time" exceeds the preset measurement time "dT"
	{
		// compute cumulated sum:
		cumsum.at(0) = (*rate_adr).at(0);
		for( int j=1; j<l_rate; j++ )				// walk along all sites
		{
			cumsum.at(j) = cumsum.at(j-1) + (*rate_adr).at(j);
		}
		lastcumsum = cumsum.at(l_rate-1);			// last entry in cumsum
		// create random number:
		randno1 = distribution(seed);				// for finding jumpx
		randno2 = distribution(seed);				// for time-update
		// get update-time to check, if jump still occurs within preset time-intervall dT:
		time += -log(randno2)/lastcumsum;			// need new random number to avoid correlation between waiting time and location
		if( time<=dT )
		{
			// identify jump (i.e. jumpx, jumptox):
			searchat = randno1*lastcumsum;
			guess = randno1*l_rate;
			rateindex = binsearch(&cumsum, searchat, max(0.,floor(guess-sqrt(guess))), min(l_rate-1.0,ceil(guess+sqrt(guess))));	// position, where particle jumps from and to, in rate
			jumpx = rateindex/4;						// location of actual particle in S, that jumps
			direction = modulo(rateindex,4);			// direction, where particle jumps: 0 for North, 1 for East, 2 for South, 3 for West
			jumptox = getneighbour(jumpx, direction, Nx, Ny, tilt, boundarycondition);
			// update state:
			spinmemory = (*S_adr).at(jumpx);
			(*S_adr).at(jumpx) = (*S_adr).at(jumptox);
			(*S_adr).at(jumptox) = spinmemory;
			// update jump-rates:
			// ...around jumpx:
			for( nndirection=0; nndirection<4; nndirection++)				// walk through nearest neighbours
			{
				nn = getneighbour(jumpx, nndirection, Nx, Ny, tilt, boundarycondition);
				if( nn<Nx || nn>=lowerseam )								// i.e. neighbour is in upper or lower boundary seam, therefore not allowed to jump
				{
					// don't do anything, as rates were initialized as zero and stay zero
				}
				else
				{
					localmean = getlocalmean(nn, S_adr, Nx, Ny, tilt, boundarycondition);
					nnenergygain = 2*(*S_adr).at(nn)*(localmean+h);			// gain of energy, if current state would be replaced by opposite spin at nn
					for( nnndirection=0; nnndirection<4; nnndirection++ )	// walk through nearest neighbours of nearest neighbours
					{
						nnn = getneighbour(nn, nnndirection, Nx, Ny, tilt, boundarycondition);	// note: jumpx is among them, so does not need to be updated separately
						backnnndirection = modulo(nnndirection+2,4);		// direction of backward-jump
						if( (*S_adr).at(nnn)==(*S_adr).at(nn) || nnn<Nx || nnn>=lowerseam )	// neighbour is the same or at upper/lower border
						{
							localrate = 0;
						}
						else												// neighbour is different
						{
							localmean = getlocalmean(nnn, S_adr, Nx, Ny, tilt, boundarycondition);
							nnnenergygain = 2*(*S_adr).at(nnn)*(localmean+h);	// gain of energy, if current state would be replaced by opposite spin at nnn
							localrate = 1./(1.+exp(b*(nnnenergygain+nnenergygain)));// rate for interchanging particles at nnn and nn
						}
						(*rate_adr).at(nn*4+nnndirection) = localrate;		// nn-->nnn
						(*rate_adr).at(nnn*4+backnnndirection) = localrate;	// nnn-->nn
					}
				}	// endif not in upper or lower boundary-seam
			}
			// ...around jumptox:
			ujumpx = getneighbour(jumpx, 0, Nx, Ny, tilt, boundarycondition);	// nn of jumpx in North-direction
			ljumpx = getneighbour(jumpx, 2, Nx, Ny, tilt, boundarycondition);	// nn of jumpx in South-direction
			for( nndirection=0; nndirection<4; nndirection++)				// walk through nearest neighbours
			{
				nn = getneighbour(jumptox, nndirection, Nx, Ny, tilt, boundarycondition);
				if( nn<Nx || nn>=lowerseam )								// i.e. neighbour is in upper or lower boundary seam, therefore not allowed to jump
				{
					// don't do anything, as rates were initialized as zero and stay zero
				}
				else if( nn!=jumpx )										// otherwise already did this update before
				{
					localmean = getlocalmean(nn, S_adr, Nx, Ny, tilt, boundarycondition);
					nnenergygain = 2*(*S_adr).at(nn)*(localmean+h);			// gain of energy, if current state would be replace by opposite spin at nn
					for( nnndirection=0; nnndirection<4; nnndirection++ )	// walk through nearest neighbours of nearest neighbours
					{	
						nnn = getneighbour(nn, nnndirection, Nx, Ny, tilt, boundarycondition);	// note: jumpx is among them, so does not need to be updated separately
						if( nnn!=jumptox && nnn!=ujumpx && nnn!=ljumpx )	// already went through nn of jumpx above, so avoid redundancy
						{
							backnnndirection = modulo(nnndirection+2,4);	// direction of backward-jump
							if( (*S_adr).at(nnn)==(*S_adr).at(nn) || nnn<Nx || nnn>=(Ny-1)*Nx  )	// neighbour is the same or at upper/lower border
							{
								localrate = 0;
							}
							else											// neighbour is different
							{
								localmean = getlocalmean(nnn, S_adr, Nx, Ny, tilt, boundarycondition);
								nnnenergygain = 2.*(*S_adr).at(nnn)*(localmean+h);	// gain of energy, if current state would be replace by opposite spin at nnn
								localrate = 1./(1.+exp(b*(nnenergygain+nnnenergygain)));// rate for interchanging particles at nnn and nn
							}
							(*rate_adr).at(nn*4+nnndirection) = localrate;		// nn-->nnn
							(*rate_adr).at(nnn*4+backnnndirection) = localrate;	// nnn-->nn
						}
					}
				}	// endif not in upper or lower boundary-seam
			}	// end of checking the neighbours
		}	// end time not beyond dT
	}	// end time-evolution
	
	return;
}