/*
	Simulates Kawasaki-type-dynamics with a Kinetic-Monte-Carlo-algorithm and periodic boundary-conditions
*/


#include <iostream>		// needed for output in control-window
#include <string>		// needed for strings/names
#include <vector>		// needed for vectors/vector-operations
#include <random>		// needed for random-number-generator

using namespace std;

#include "binsearch.h"
#include "modulo.h"		// for periodic boundary-conditions
#include "getjumprates.h"

void Kawasaki_KMC( vector<int>* Eta_new_adr, vector<double>* rate_adr, double dT, int sitenumber, int d, double beta )
{
	double time = 0;					// initial physical time
	vector<double> cumsum(2*d*sitenumber);// cumulated sum of jump-rates of the sites in order of rate (i.e. 1left, 1right, 2left, 2right,...)
	double lastcumsum = 0;				// last element of cumsum (i.e. sum over all jump-rates)

	// initialize random distribution:
	random_device rd;
	mt19937_64 seed(rd());
	uniform_real_distribution<double> distribution(0.0,1.0);
	double randno1 = 0, randno2 = 0;	// initializes random number between 0 and 1
	double randno3 = 0;
	int direction_rate = -1;			// initialize variable, that decides if to jump right or left
	int jumpdim = -1, jumpdir = 0;		// dimension in which the jump takes place; direction in which the jump takes place (+-1)
	int sitenumber_dim = pow(sitenumber,1./double(d));	// number of sites in each direction

	// initialize loop-variables:
	int jumpx = -1, jumptox = -1, nn = -1;
	double searchat = 0.0, guess = -1.;
	double avdiffatjump, avdiffatnn;

	while( time <= dT )													// run until physical time "time" exceeds the preset measurement time "dT"
	{
		// compute cumulated sum:
		cumsum.at(0) = (*rate_adr).at(0);
		for( int j=1; j<2*sitenumber; j++ )								// walk along all sites
		{
			cumsum.at(j) = cumsum.at(j-1) + (*rate_adr).at(j);
		}
		lastcumsum = cumsum.at(2*d*sitenumber-1);						// last entry in cumsum
		// create random number:
		randno1 = distribution(seed);									// for finding jumpx
		randno2 = distribution(seed);									// for time-update
		// identify jump (i.e. jumpx, jumptox) and update time:
		searchat = randno1*lastcumsum;
		guess = randno1*2*d*sitenumber;									// 2*d possible neighbours to jumpt to
		jumpx = binsearch(&cumsum, searchat, max(0.,floor(guess-sqrt(guess))), min(2*d*sitenumber-1.0,ceil(guess+sqrt(guess))));	// position, where particle jumps from (to left or right)
		int Jumpx = jumpx;
		direction_rate = modulo(jumpx,2*d);								// "0" for left-jump, "1" for right-jump
		jumpdim = int(direction_rate/2);								// dimension 0..(d-1) in which the jump takes place
		jumpdir = 2*modulo(direction_rate,2)-1;							// "+1" for jump to the right, "-1" for jump to the left
		jumpx = jumpx/2;												// gets rounded down automatically by "int"
		time += -log(randno2)/lastcumsum;								// need new random number to avoid correlation between waiting time and location
		if( time<=dT )													// don't jump, if new time is beyond terminal time
		{
			jumptox = jumpx - modulo(jumpx,pow(sitenumber_dim,double(jumpdim+1))) + modulo( jumpx+jumpdir*pow(sitenumber_dim,double(jumpdim)), pow(sitenumber_dim,double(jumpdim+1)));	// with periodic boundary-conditions
			// update state:
			(*Eta_new_adr).at(jumpx) += -1;								// particle jumps away from jumpx
			(*Eta_new_adr).at(jumptox) += 1;							// particle jumped to "jumptox"
			// update rate:
			// ...first around jumpx:
			getjumprates( Eta_new_adr, rate_adr, jumpx, sitenumber_dim, d, beta);
			// ...then around jumptox:
			getjumprates( Eta_new_adr, rate_adr, jumptox, sitenumber_dim, d, beta);
		}	// end time not beyond dT
	}	// end time-evolution

	return;
}