/*
	Binary search of value in an ascendingly ordered double-list; 
	first try to find within the range [lowstarguess, highstartguess]
	returns list-index of nearest greater list-value or "-1" in case larger than all list-entries

*/

#include <iostream>		// needed for output in control-window
#include <vector>		// needed for vectors/vector-operations

using namespace std;

int binsearch( vector<double>* list_adr, double value, int lowstartguess, int highstartguess )
{
	int newpos = -1;						// initialize returnvalue impossibly to indicate it wasn't found
	int l = (*list_adr).size();				// length of list

	// return 0, if list is empty:
	if( l==0 )
	{
		return -1;
	}

	// check if low-/highstartguesses are reasonable:
	if( lowstartguess > highstartguess )
	{
		cout << "  Warning - binsearch: Initial guesses in wrong order - swap them instead\n";
		int tempstore = lowstartguess;
		lowstartguess = highstartguess; highstartguess = tempstore;
	}
	if( lowstartguess >= l || lowstartguess < 0 )
	{
		cout << "  Warning - binsearch: Initial lower guess out of indexrange - choose default instead\n";
		lowstartguess = ceil(double((l-1)/3));
	}
	if( highstartguess >= l || highstartguess < 0 )
	{
		cout << "  Warning - binsearch: Initial upper guess out of indexrange - choose default instead\n";
		highstartguess = ceil(double(2*(l-1)/3));
	}
	if( lowstartguess > highstartguess )
	{
		cout << "  Info - binsearch: New guesses in wrong order - swap them instead\n";
		int tempstore = lowstartguess;
		lowstartguess = highstartguess; highstartguess = tempstore;
	}

	// initialize parameters:
	int bestlowestguess = 0;			// lower bound of index in list
	int besthighestguess = l-1;			// upper bound of index in list
	int currentguess = floor( double( (lowstartguess+highstartguess)/2 ) );	// current guess of index in list

	// check boundaries: (do this right-away, so we don't have to check in the loop later; could also do right after realizing, that startguesses are off for performance, but less nice)
	if( (*list_adr).at(bestlowestguess) >= value )
	{
		return 0;						// i.e. beyond the list on lower side or equal to lowest value
	}
	if( (*list_adr).at(besthighestguess) < value )
	{
		return -1;						// i.e. beyond the list on upper side
	}
	if( (*list_adr).at(besthighestguess) == value )
	{
		return besthighestguess;		// i.e. already found it at upper end of list
	}

	// check initial guesses:
	if( (*list_adr).at(lowstartguess) <= value )
	{
		bestlowestguess = lowstartguess;
		if( (*list_adr).at(highstartguess) >= value )
		{
			besthighestguess = highstartguess;
		}
	}
	else								// initial lower guess was too high ==> besthighestguess
	{
		besthighestguess = lowstartguess;
	}

	// run binary search iteration:
	while( newpos == -1 )
	{
		if( (*list_adr).at(currentguess) > value )
		{
			besthighestguess = currentguess;
			currentguess = floor( double( (bestlowestguess+currentguess)/2 ) );
		}
		else if( (*list_adr).at(currentguess) < value )
		{
			bestlowestguess = currentguess;
			currentguess = floor( double( (currentguess+besthighestguess)/2 ) );
		}
		else
		{
			newpos = currentguess;							// found the index
		}
		if( besthighestguess-bestlowestguess == 1 )			// would be stuck, if just run the algorithm above
		{
			newpos = besthighestguess;						// ==> rather go for the larger one, if between two values
		}
		else if( besthighestguess-bestlowestguess == 0 )	// should already be excluded by initial check of boundaries in combination with == value above (but still occurs for value = -1.#IND)
		{
			if( besthighestguess == 0 )
			{
				newpos = -1;
			}
			else											// i.e. bestlowestguess == l-1
			{
				newpos = l;									// i.e. out of bounds
			}
		}
	}

	return newpos;
}