/*
	Controls the time-evolution-, state-saving- and sigma-evaluation-processes
*/

#include <iostream>		// needed for output in control-window
#include <ctime>		// needed for timer
#include <string>		// needed for strings/names
#include <sstream>		// needed for ostringstream
#include <vector>		// needed for vectors/vector-operations
#include <numeric>		// needed for accumulate or innerproduct
#include <algorithm>	// needed for copy, max, max_element
#include <fstream>		// needed for output to external files
#include <math.h>		// needed for sqrt

using namespace std;

#include "getxrange.h"
#include "getrho_old.h"
#include "geteta_old.h"
#include "microtomacro.h"
#include "microtomacro_double.h"
#include "zero_range_KMC.h"
#include "getoriginindex.h"
#include "zero_range_KMC_multitag.h"
#include "getMSD.h"
#include "simple_exclusion_KMC.h"
#include "Kawasaki_KMC.h"
#include "Kawasaki_Ising_KMC.h"
#include "Findloop.h"
#include "converttoheight.h"
#include "savetotext.h"
#include "binsearch.h"
#include "savestatetotext.h"
#include "gamma.h"
#include "getY.h"
#include "getsigma.h"
#include "getrho_a.h"
#include "getF.h"
#include "modulo.h"
#include "saveresultstotext.h"
#include "interpolate.h"
#include "getneighbour.h"
#include "getlocalmean.h"
#include "getfilename.h"

void control( double dT, int Nbin, int micres, int Npart, double wait, double fwait, double x0range[], const int lx0range, double refinement, int samplesize, int storagelength, string name, string samplingmode,  int sampling, int fsampling, string type, string mode, string repeattype, string profile, string filtertype, vector<double>* filterparameters_adr, string interpolationtype, int interpolationref, string initializationtype )
{
	// for timer-output:
	clock_t t1 = clock();						// start-time of timer

	// to avoid bad surprises:
	if( type!="ZRP" && type!="SEP" && type!="ZRP_multitag" && type!="Kawasaki" && type!="Kawasaki_Ising" )
	{
		cout << "  Warning - control: No valid process-type chosen: " << type << ". Abort the programme.\n";
		return;
	}

	int timesample_no = 100;
	double timesample_time = 0.5*dT/pow(timesample_no,1.), timesample_forwardshift = 0;
	if( mode == "repeat" && repeattype != "keepinitial"  && repeattype != "localaverage" && repeattype != "purelystochastic" && repeattype != "timesample" && repeattype != "statesample" )
	{
		cout << " Warning - control: Unkown repeattype: " << repeattype << "\n";
		return;
	}
	else if( mode == "repeat" && repeattype == "timesample" )
	{
		timesample_forwardshift = timesample_time*double(timesample_no)/2.;
		// change waiting/measurement times intermittently to keep same code in the next parts as for other repeattypes:
		wait = wait - timesample_forwardshift;	// shorten waiting time a little bit, such that several samples can be taken around the actual waiting time, to average them to estimate deterministic rho_old
		if( wait<0 )
		{
			wait = 0;
			cout << " Warning - control: Too short waiting time for repeattype '" << repeattype << "', set to " << timesample_forwardshift << " instead.\n";
		}
		dT = dT - timesample_forwardshift;		// shorten waiting time a little bit, such that several samples can be taken around the actual measurement time, to average them to estimate deterministic rho_new
		if( dT<0 )
		{
			dT = 0;
			cout << " Warning - control: Too short measurement time for repeattype '" << repeattype << "', set to " << timesample_forwardshift << " instead.\n";
		}
		// change storagelength, as now also deterministic states have to be saved:
		storagelength = max(4,storagelength*2);		// need to store deterministic and stochastic states pairwise (first eta_new, then rho_new; initially also: eta_old, rho_old)
		
		if( fwait>0 )
		{
			cout << " Warning - control: Bad combination repeattype '" << repeattype << "' and fwait = " << fwait << ", set fwait = 0.\n";
			fwait = 0;
		}
		if( fsampling!=1 )
		{
			cout << " Warning - control: Bad combination repeattype '" << repeattype << "' and fsampling = " << fsampling << ", set fsampling = 1.\n";
			fsampling = 1;
		}
	}
	else if( mode == "repeat" && repeattype == "statesample" )
	{
		if( samplesize!=1 )
		{
			cout << " Warning - control: Samplesize of " << samplesize << " reset to 1 for repeattype '" << repeattype << "'.\n";
			samplesize = 1;
		}
	}
	vector<double> storage(Nbin*storagelength);	// set of all eta_new until next saving in external file
	if( fsampling != 1 )
	{
		if( mode == "repeat" && ( repeattype == "keepinitial" || repeattype == "timesample" || repeattype == "localaverage") )
		{
			cout << " Warning - control: Bad combination repeattype '" << repeattype << "' and fsampling = " << fsampling << ", set fsampling = 1.\n";
			fsampling = 1;
		}
	}

	// initialize variables:
	int j_overallsampling = 0;					// j_sampling + j_fsampling
	int overallsampling;
	if( samplingmode == "combined" )
	{
		overallsampling = sampling*fsampling;
	}
	else if( samplingmode == "separate" )
	{
		overallsampling = fsampling;
	}
	else
	{
		cout << " Warning - control: Unexpected samplingmode: '" << samplingmode << "' set to 'separate'.\n"; 
		samplingmode = "separate";
		overallsampling = fsampling;
	}
	int totalsample = overallsampling*samplesize;
	int Nmic = Nbin*micres;						// total amount of microscopical sites in 1D
	int Npart_init = Npart;						// initial Npart (needed for repeated samplings of non-flat profiles, as otherwise initial information would be lost)
	double dT_init = dT, wait_init = wait;		// ...similarly for dT and wait
	int Npart_filename = Npart;					// particle number used in filename (in case it slightly changes between samplings for stochastic reasons)
	int softener = 1;
	int Nint = Nbin*interpolationref, Nsoft = Nbin/softener;
	double dt = dT/pow(Nmic,2.);				// macroscopic time
	vector<double> xrange(Nint);				// discrete space
	vector<double> boundaries(2);				// initialize ends of domain xrange
	boundaries.at(0) = 0; boundaries.at(1) = 1;	// i.e. [0,1]
	vector<double> Rho_old(Nmic);				// deterministic (microscopical) initial distribution
	vector<double> rho_old(Nbin);				// deterministic initial distribution (only macroscopical resolution needed for the evaluation)
	vector<double> rho_old_int(Nint);			// interpolated rho_old
	vector<double> rho_old_soft(Nsoft);			// softened version of rho_old(_int) by local
	vector<double> rho_new(Nbin);				// deterministic final distribution (only macroscopical resolution needed for the evaluation)
	vector<double> rho_new_int(Nint);			// interpolated rho_new
	vector<double> rho_new_soft(Nsoft);			// softened version of rho_new(_int) by local averaging
	vector<int> Eta_old(Nmic);					// particle-distribution (based on Rho_old and used as basis for evolution in stochastic processes)
	vector<int> fEta_old(Nmic);					// particle-distribution after wait - fwait (i.e. processed in fsamplings)
	vector<double> Eta_old2(Nmic);				// microscopical intitial distribution (as double)
	vector<double> eta_old(Nbin);				// macroscopical intitial distribution
	vector<double> eta_old_int(Nint);			// interpolated eta_old
	vector<int> Eta_new(Nmic);					// particle-distribution some time after Eta_old (used inside stochastic processes)
	vector<double> eta_new(Nbin);				// macroscopical final distribution
	vector<double> eta_new_int(Nint);			// interpolated eta_new
	vector<double> gamma_im(Nint);				// image of xrange over gamma (ordered accordingly)
	vector<double> allgamma_im(Nint*lx0range);	// image of xrange over gamma for all x0 in x0range, one after another
	vector<double> allgammasq_im(Nint*lx0range);// derivative of allgamma_im
	vector<double>::iterator gamma_start;		// address of first element in allgamma_im for certain x0
	vector<double>::iterator gammasq_start;		// address of first element in allgammasq_im for certain x0
	vector<double> drhodT(lx0range);			// d rho/dt = (rho_new-eta_old)/dt measured by gamma centered at respective x0
	vector<double> sigmasample(totalsample*lx0range);		// storage for the sigmas of all the individual samples (store all sample-points of one x0 in a row, and then all sample-points (sample, fsampling and sampling in this order) of another x0 after it)
	vector<double> diametentsample(totalsample*lx0range);	// storage for the diagonal metric metries of all the individual entries
	vector<double> eta_a(lx0range);				// eta_old*gs.^2
	vector<double> eta_a_a(lx0range);			// average over all samplings of eta_a
	vector<double> rho_a(lx0range);				// rho_new*gs.^2
	vector<double> rho_a_a(lx0range);			// average over all samplings of rho_a
	int J = 0;									// current number of states stored in storage (i.e. either number of eta_news or four states in case of timesample)
	int j_sigmasample = 0;						// number of samplings already stored in sigmasample
	int samplepos = -1, nextsamplepos = -1;		// for index within sigmasample and diametentsample
	vector<int> empty;							// empty vector (needed for some of the processes if not multispecies-case is studied)
	double Y_new = 0, Y_old = 0, sigma = -1, diametent = -1;// initialize parameters for later
	vector<double> average, stdderr;			// list of results for sigma and its error at each point x0
	vector<double> avdiametent, stderr_diametent;	// list of result for diagonal metric entries and its standard deviation
	string suffix, filename;
	stringstream suffixstream;
	bool withintro = 1;							// ==1 if withintro, ==0 if without intro in savestatetotext-file

	double epsilon = 1;
	
	// for multi-particle-tracking:
	vector< vector<int> > originindex(Nmic);	// needed for multiple-particle-tracking, initialize with as many vector-lists as positions in the microscopical domain
	vector< vector<int> > originindex_memory(Nmic);
	vector<double> MSD;							// list of Mean-Square-Displacements in case of Multitag

	// for Kawasaki(_Ising):
	vector<double> rate;						// jumprates for Kawasaki(_Ising)-dynamics (size will be initialised later, if needed)
	vector<double> rate_old;					// jumprates for Kawasaki(_Ising)-dynamics for original state Eta_old just after preinitial waiting time
	vector<double> frate_old;					// jumprates for Kawasaki(_Ising)-dynamics for original state just after fwait

	// for Ising:
	double tilt = 0./Nmic;						// tilt
	int Ny, Ntot;								// vertical size of 2D Ising with Nx = Nmic; total number of pixels in 2D Ising
	double h = 0., T = 0.2, b = 1./T;			// bias/ external force; temperature; inverse temperature
	vector<int> S, S_old, fS_old;				// spin-matrix
	Findloop surface;							// surface between -1 and +1 spins
	int lastlength = Nmic;						// approximate length of surface
	vector<int> startpt(2);						// starting point for search
	vector<int> startavoid;						// initialize empty startavoid (points to avoid initially)
	string boundarycondition = "";				// blank space to set boundary condition (currently always set to periodic in x-direction and constant +/- in y-direction)

	// for wait/ fwait:
	wait = wait - fwait;						// reset "wait" such that overall waiting time until initial state is "wait"
	if( wait<0 )
	{
		cout << " Warning - control: Bad waiting time " << wait + fwait << " in combination with fwait of " << fwait << " therefore changed to " << fwait << ".\n";
		wait = 0;								// need at least (real) waiting time of fwait to allow for preinitial waiting of fwait
	}

	// get space:
	getxrange(&xrange,Nint,boundaries.at(0),boundaries.at(1));	// (discrete) space
	double dx = xrange.at(1) - xrange.at(0);	// space-increment
	double Dx = dx/refinement;					// microscopical increment for determining better resolution in derivatives (nabla gamma)
	double x0 = -1;								// x0-value, where filter-function gamma is concentrated
	
	// preset allgamma and allgammasq for all x0:
	for( int j_x0=0; j_x0<lx0range; j_x0++ )		// go through all x0 in range
	{
		x0 = x0range[j_x0];							// set current x0
		gamma(&gamma_im, &xrange, Nint, x0, filtertype, filterparameters_adr);	// get gamma_im for particular x0
		copy(gamma_im.begin(),gamma_im.end(),allgamma_im.begin()+j_x0*Nint);	// copy gamma_im into respective position of allgamma_im
	}

	vector<double> neighbours(2);					// initialize the two neighbours for the points in the domain
	vector<double> loc_gamma_im(2);					// initialize gamma(neighbours)
	for( int j_sq=0; j_sq<Nint*lx0range; j_sq++ )	// walk along space
	{
		if( refinement == 1 )
		{
			allgammasq_im.at(j_sq) = ( allgamma_im.at(j_sq-modulo(j_sq,Nint)+modulo(j_sq+1,Nint))-allgamma_im.at(j_sq-modulo(j_sq,Nint)+modulo(j_sq-1,Nint)) )/(2*dx);	// Euler-approx. for derivative (modulo for periodic boundary-conditions (otherwise not close to the boundary anyways))
		}
		else
		{
			neighbours.at(0) = xrange.at(j_sq)-Dx; neighbours.at(1) = xrange.at(j_sq)+Dx;	// update neighbours (might be negative, too; this does not work, if gamma is not zero on the boundary and in a neighbourhood)
			gamma( &loc_gamma_im, &neighbours, 2, x0, filtertype, filterparameters_adr );	// update loc_gamma_im
			allgammasq_im.at(j_sq) = ( loc_gamma_im.at(1)-loc_gamma_im.at(0) )/(2.*Dx);		// Euler-approximation for derivative
		}
		allgammasq_im.at(j_sq) = pow( allgammasq_im.at(j_sq), 2.0 );// square
	}
		
	if( interpolationtype=="const" && interpolationref!=1 )			// enough to interpolate allgamma, allgammasq, don't need to "refine" eta_old, eta_new 
	{
		for( int j_nonref=0; j_nonref<Nbin*lx0range; j_nonref++ )	// walk along contracted allgamma, allgammsq
		{
			allgamma_im.at(j_nonref) = accumulate(allgamma_im.begin()+j_nonref*interpolationref,allgamma_im.begin()+(j_nonref+1)*interpolationref,0.)/interpolationref;		// take average over interpolationref many entries and map to corresponding position in Nmic large allgamma_im
			allgammasq_im.at(j_nonref) = accumulate(allgammasq_im.begin()+j_nonref*interpolationref,allgammasq_im.begin()+(j_nonref+1)*interpolationref,0.)/interpolationref;
		}
		allgamma_im.resize(Nbin*lx0range);				// only keep the nonrefined length of allgamma_im (with the averaged values of the whole vector before) and discard the rest
		allgammasq_im.resize(Nbin*lx0range);
		// also shrink xrange to "real" size:
		xrange.resize(Nbin);							// discrete space
		getxrange(&xrange,Nbin,boundaries.at(0),boundaries.at(1));	// (discrete) space
		dx = xrange.at(1) - xrange.at(0);				// space-increment
		Dx = dx/refinement;								// microscopical increment for determining better resolution in derivatives (nabla gamma)
		// also shrink "interpolated" states, as they don't need to be interpolated anymore:
		eta_old_int.resize(Nbin);
		rho_old_int.resize(Nbin);
		eta_new_int.resize(Nbin);
		rho_new_int.resize(Nbin);
		// reset sitenumber to Nbin rather than Nint to allow to use same formulas below:
		Nint = Nbin;
		interpolationref = 1;							// make sure the states don't get refined any further
	}
	
	dT_init = dT; wait_init = wait;						// save current measurement/waiting time as initial times for each sampling
	// walk along samplings:
	for( int j_sampling=0; j_sampling<sampling; j_sampling++ )
	{
		J = 0;											// reset inner-storage-index
		Npart = Npart_init;								// reset initial particle-number to make sure, same situation is recreated in each sampling
		dT = dT_init; wait = wait_init;					// ...similarly for measurement/waiting times
		clock_t t2 = clock();							// start-time of timer

		// create pre-initial state:
		if( withintro )
		{
			if( samplingmode == "combined" )			// otherwise missleading as j_sampling==0 always inside control
			{
				cout << " (" << j_overallsampling+1 << ")\n";
			}
			cout << "_____Npart = " << Npart;			// preset particlenumber
		}
		getrho_old(&Rho_old,Nmic,Npart,profile);			// deterministic (pre-) initial state
		getEta_old(&Eta_old,&Rho_old,Nbin,micres,Npart, type, initializationtype);	// particle-distribution based on deterministic rho_old
		Npart = accumulate(Eta_old.begin(),Eta_old.end(),0);// update particlenumber
		if( j_overallsampling==0 )
		{
			Npart_filename = Npart;							// Npart of first sampling is taken as reference particle-number for name
		}
		if( withintro )
		{
			cout << " --> particlenumber = " << Npart;		// actual particlenumber
		}

		// wait pre-pre-initial waiting time to get pre-initial state:
		if( type == "ZRP" )
		{
			average.resize(lx0range); stdderr.resize(lx0range);
			avdiametent.resize(lx0range); stderr_diametent.resize(lx0range);
			zero_range_KMC(&Eta_old, &empty, wait, Nmic);	// wait initial waiting time to get beyond initial layer
		}
		else if( type == "ZRP_multitag" )
		{
			MSD.resize(totalsample);
			zero_range_KMC(&Eta_old, &empty, wait, Nmic);
		}
		else if( type == "SEP" )
		{
			average.resize(lx0range); stdderr.resize(lx0range);
			avdiametent.resize(lx0range); stderr_diametent.resize(lx0range);
			simple_exclusion_KMC(&Eta_old, wait, Nmic, Npart);	// wait initial waiting time to get beyond initial layer
		}
		else if( type == "Kawasaki" )
		{
			average.resize(lx0range); stdderr.resize(lx0range);
			avdiametent.resize(lx0range); stderr_diametent.resize(lx0range);
			// initialize rates:
			rate.resize(2*Nmic);
			rate_old.resize(2*Nmic);				// resize original rates accordingly
			frate_old.resize(2*Nmic);
			int direction_rate = -1, jumpx = -1, jumptox = -1;
			double diffprob = -1;
			for( int j_rate=0; j_rate<2*Nmic; j_rate++ )
			{
				direction_rate = modulo(j_rate,2);	// 0 for left-jump, 1 for right-jump
				jumpx = j_rate/2;					// position in xrange, Eta_old etc; gets rounded down by "int"
				jumptox = modulo(jumpx+(2*direction_rate-1), Nmic);
				if( Eta_old.at(jumpx) != 0 )		// only if particle exists at x_j_rate, otherwise, don't do anything
				{
					diffprob = exp( 2.*(Eta_old.at(jumpx)-Eta_old.at(jumptox)) - 2. );
					rate.at(j_rate) = diffprob/(diffprob+1.);	// jumpx -> jumptox
				}
				else
				{
					rate.at(j_rate) = 0;
					j_rate++;						// can skip next one (i.e. corresponding jump to the right) as is 0 as well
					rate.at(j_rate) = 0;
				}
			}
			Kawasaki_KMC(&Eta_old, &rate, wait, Nmic);
			if( !(mode == "repeat" && repeattype == "timesample") )	// otherwise not yet at end of real waiting time
			{
				rate_old = rate;
			}
		}
		else if( type == "Kawasaki_Ising" )
		{
			average.resize(lx0range); stdderr.resize(lx0range);
			avdiametent.resize(lx0range); stderr_diametent.resize(lx0range);
			// update name-suffix to include T and h:
			stringstream newsuffix;
			newsuffix << "T_=_" << T << ",_h_=_" << h << ",_" << suffix;
			suffix = newsuffix.str();
			// initialize 2D-variables:
			Ny = max( ceil(6.*sqrt(double(Nmic)))+2., double( ceil( *max_element(Eta_old.begin(),Eta_old.end()) + 6.*sqrt(double(*max_element(Eta_old.begin(),Eta_old.end())))) +2. ));
			Ntot = Ny*Nmic;
			S.resize(Ntot); S_old.resize(Ntot);
			double localmean, energygain, nnenergygain, localrate;
			int y, x, nn, backnndirection;
			// initialize state S:
			for( int j_S=0; j_S<Ntot; j_S++ )	// walk through physical space
			{
				y = Ny - j_S/Nmic - 1;			// row-position counted from the bottom starting from 0
				x = modulo(j_S,Nmic);			// column-position counted from the left starting from 0
				if( y<=Eta_old.at(x) +2*modulo(x,2)-1 && y<Ny-1 && y>0 )
				{
					S.at(j_S) = +1;				// +1 below Eta_old-curve
				}
				else if( y>0 )
				{
					S.at(j_S) = -1;				// -1 above Eta_old-curve
				}
				else
				{
					S.at(j_S) = +1;				// bottom seam (in case Eta_old would be too low)
				}
			}
			// initialize rates:
			rate.resize(4*Ntot,0.);				// zero, except if otherwise stated below
			rate_old.resize(4*Ntot,0.);			// resize memory for origininal rates accordingly
			frate_old.resize(4*Ntot,0.);
			for( int j_S=Nmic; j_S<(Ny-1)*Nmic; j_S++)	// walk along all pixels of 2D-space
			{
				localmean = getlocalmean(j_S, &S, Nmic, Ny, tilt, boundarycondition);
				energygain = 2.*S.at(j_S)*(localmean+h);				// gain of energy, if current state would be replace by opposite spin at jumpx
				for( int nndirection=1; nndirection<3; nndirection++ )	// only need to update East and South position for no redundancy
				{
					nn = getneighbour(j_S, nndirection, Nmic, Ny, tilt, boundarycondition);	// note: nn might be beyond physical indexrange of S, but these cases don't get processed due to "if" below
					if ( S.at(nn)!=S.at(j_S) && nn>=Nmic && nn<(Ny-1)*Nmic )	// i.e. jump may occur
					{
						backnndirection = modulo(nndirection+2,4);		// direction of backward-jump
						localmean = getlocalmean(nn, &S, Nmic, Ny, tilt, boundarycondition);
						nnenergygain = 2.*S.at(nn)*(localmean+h);		// gain of energy, if current state would be replaced by opposite spin at nnx
						localrate = 1./(1.+exp(b*(energygain+nnenergygain)));// rate for interchanging particles at jumptox and nn
						rate.at(j_S*4+nndirection) = localrate;			// j_S-->nnx
						rate.at(nn*4+backnndirection) = localrate;		// nnx-->j_S
					}
				}
			}
			// wait:
			Kawasaki_Ising_KMC( &S, &rate, 4*Ntot, wait, Nmic, Ny, tilt, b, h, boundarycondition);
			if( !(mode == "repeat" && repeattype == "timesample") )		// otherwise not yet at end of real waiting time
			{
				savetotext(Ny, Nmic, tilt, T, h, 0, wait, 0, &S);
			}
		}
		copy(Eta_old.begin(),Eta_old.end(),fEta_old.begin());			// set fEta_old;

		// start pre-initial sampling-process (for getting detrministic rho_old by statesampling or just fluctuation-sampling):
		for( int j_fsampling=0; j_fsampling<fsampling; j_fsampling++ )
		{
			J = 0;														// reset inner-storage index
			//clock_t t2 = clock();										// start-time of timer
			withintro = (j_overallsampling==0)||(mode == "continuous")||(repeattype != "statesample"); // ==1 if withintro, ==0 if without intro
			copy(fEta_old.begin(),fEta_old.end(),Eta_old.begin());		// reset Eta_old to equal fEta_old;
			dT = dT_init; wait = wait_init;								// get reset, when file gets read again


			// create correct suffix:
			if( samplingmode == "separate" )
			{
				suffix = name;								// suffix already created in main
			}
			else if( samplingmode == "combined" )
			{
				suffixstream.str("");						// clear earlier content of the name-stringstream
				if( mode == "repeat" && repeattype == "statesample" )	// i.e. statesample is active (want same file/filename for all data)
				{
					suffixstream << filtertype << "_" << interpolationtype << "(" << interpolationref << ")" << "_" << profile  << "_" << name << "_1-" << overallsampling;
				}
				else
				{
					suffixstream << filtertype << "_" << interpolationtype << "(" << interpolationref << ")" << "_" << profile << "_" << name << "_" << j_overallsampling+1;
				}
				suffix = suffixstream.str();
			}
			if( mode == "repeat" && repeattype == "timesample" )
			{
				// update name-suffix to include T and h:
				stringstream newsuffix;
				newsuffix << "sno_=_" << timesample_no << ",_sti_=_" << timesample_time << ",_" << suffix;
				suffix = newsuffix.str();
			}

			// wait pre-initial waiting time of fwait:
			if( type == "ZRP" )
			{
				zero_range_KMC(&Eta_old, &empty, fwait, Nmic);			// wait fwait
			}
			else if( type == "ZRP_multitag" )
			{
				zero_range_KMC(&Eta_old, &empty, fwait, Nmic);
				getoriginindex( &Eta_old, Nmic, &originindex );			// generate list of original positions of particles
				originindex_memory = originindex;						// make sure to remember originindex
			}
			else if( type == "SEP" )
			{
				simple_exclusion_KMC(&Eta_old, fwait, Nmic, Npart);		// wait fwait
			}
			else if( type == "Kawasaki" )
			{
				rate = rate_old;										// reset rate to state at end of waiting period
				Kawasaki_KMC(&Eta_old, &rate, fwait, Nmic);
				frate_old = rate;										// save new rates for possibly multiple samples
			}
			else if( type == "Kawasaki_Ising" )
			{
				S = S_old; rate = rate_old;								// recreate situation from just after waiting period
				Kawasaki_Ising_KMC( &S, &rate, 4*Ntot, fwait, Nmic, Ny, tilt, b, h, boundarycondition);
				savetotext(Ny, Nmic, tilt, T, h, 0, wait, 0, &S);
				fS_old = S;	frate_old = rate;							// save original state - S, rate - for later reference
				// initialize startpoints:
				startpt.at(0) = (Ny-Eta_old.at(0))*Nmic;				// expect startpoint, where Eta_old started from
				startpt.at(1) = startpt.at(0)-Nmic;						// one row above (i.e. first -1)
				// get surface:
				surface.searchstart( &startpt, &S, Ny, Nmic, tilt, &startavoid, &startavoid, lastlength);	// extracts surface
				converttoheight( &Eta_old, surface.getloop(), surface.getnonred_amount(), surface.getloopo(), surface.getnonred_amounto(), &S, Nmic, Ny );	// avoid overhangs to get height instead of surface
				startpt.at(0) = (*surface.getloop_path()).at(0); startpt.at(1) = (*surface.getloopo_path()).at(0);
			}	// end if type
			
			// process old state:
			if( mode == "continuous" || repeattype == "keepinitial" || repeattype == "localaverage" || repeattype == "purelystochastic" || repeattype == "statesample" )
			{
				Npart = accumulate(Eta_old.begin(),Eta_old.end(),0);		// update particlenumber
				if( withintro )
				{
					cout << " --> particlenumber = " << Npart << "_____\n";	// actual particlenumber
				}
				copy(Eta_old.begin(),Eta_old.end(),Eta_old2.begin());		// get "macroscopical" eta_old, i.e. recast as double for savetext
				savestatetotext(&Eta_old2, dT, &boundaries, Nbin, micres, Npart_filename, sampling, fsampling, samplesize, wait+fwait, fwait, type, mode, repeattype, suffix, withintro );	// initialize new file and save in first place in there (need to do this after Npart-update for consistency)
				microtomacro(&Eta_old,&eta_old,Nbin,micres);				// get truely macroscopical eta_old for the final evaluation in getsigma
				interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_old for later evaluation
			
				if( repeattype == "localaverage" )
				{
					// get softened rho_old for evaluation of Y_old:
					microtomacro(&Eta_old,&rho_old_soft, Nsoft,micres*softener);
					interpolate(&rho_old_soft, &rho_old_int, Nsoft, interpolationref*softener,"const");
					for( int j_soft=0; j_soft<Nint; j_soft++ )
					{
						rho_old_int.at(j_soft) = rho_old_int.at(j_soft)*softener;
					}
				}
				else if( repeattype == "statesample" )
				{
					// get deterministic rho_old by averaging (over fsampling and sampling):
					for( int j_rho=0; j_rho<Nbin; j_rho++ )
					{
						rho_old.at(j_rho) = ( rho_old.at(j_rho)*(j_overallsampling) + eta_old.at(j_rho) )/(j_overallsampling+1);
					}
				}
			}
			else if( repeattype == "timesample" )
			{
				microtomacro(&Eta_old,&rho_old,Nbin,micres);				// initialize rho_old
				for( int j_ts=0; j_ts<timesample_no; j_ts++ )
				{
					// run process timesample_no often for very short timesample_time to then average over these time-local states to estimate deterministic rho_old:
					if( type == "ZRP" )
					{
						zero_range_KMC(&Eta_old, &empty, timesample_time, Nmic);		// wait intersample time
					}
					else if( type == "ZRP_multitag" )
					{
						cout << " Warning - control: Bad combination: " << type << " and " << repeattype << ".\n";
						return;
					}
					else if( type == "SEP" )
					{
						simple_exclusion_KMC(&Eta_old, timesample_time, Nmic, Npart);	// wait intersample time
					}
					else if( type == "Kawasaki" )
					{
						Kawasaki_KMC(&Eta_old, &rate, timesample_time, Nmic);
						if( j_ts==timesample_no/2 )							// real waiting time done
						{
							frate_old = rate;
						}
					}
					else if( type == "Kawasaki_Ising" )
					{
						Kawasaki_Ising_KMC( &S, &rate, 4*Ntot, timesample_time, Nmic, Ny, tilt, b, h, boundarycondition);
						if( j_ts==timesample_no/2 )							// real waiting time done
						{
							savetotext(Ny, Nmic, tilt, T, h, 0, wait, 0, &S);
							fS_old = S;	frate_old = rate;					// save original state - S, rate - for later reference
							// initialize startpoints:
							startpt.at(0) = (Ny-Eta_old.at(0))*Nmic;		// expect startpoint, where Eta_old started from
							startpt.at(1) = startpt.at(0)-Nmic;				// one row above (i.e. first -1)
						}
						// get surface:
						surface.searchstart( &startpt, &S, Ny, Nmic, tilt, &startavoid, &startavoid, lastlength);	// extracts surface
						converttoheight( &Eta_old, surface.getloop(), surface.getnonred_amount(), surface.getloopo(), surface.getnonred_amounto(), &S, Nmic, Ny );	// avoid overhangs to get height instead of surface
						startpt.at(0) = (*surface.getloop_path()).at(0); startpt.at(1) = (*surface.getloopo_path()).at(0);
					}	// end if type
					// update rho_old:
					microtomacro(&Eta_old,&eta_old,Nbin,micres);			// get macroscopical eta_old for the final evaluation in getsigma
					for( int j_rho=0; j_rho<Nbin; j_rho++ )
					{
						rho_old.at(j_rho) = ( rho_old.at(j_rho)*(j_ts+1) + eta_old.at(j_rho) )/(j_ts+2);
					}
					// get eta_old, eta_old_int at half-time:
					if( j_ts==timesample_no/2 )								// real waiting time done
					{
						Npart = accumulate(Eta_old.begin(),Eta_old.end(),0);// update particlenumber
						if( withintro )
						{
							cout << " --> particlenumber = " << Npart << "_____\n";	// actual particlenumber
						}
						copy(Eta_old.begin(),Eta_old.end(),Eta_old2.begin());	// get "macroscopical" eta_old, i.e. recast as double for savetext
						savestatetotext(&Eta_old2, dT+timesample_forwardshift, &boundaries, Nbin, micres, Npart_filename, sampling, fsampling, samplesize, wait+fwait+timesample_forwardshift, fwait, type, mode, repeattype, suffix, 1 );	// initialize new file and save in first place in there (need to do this after Npart-update for consistency)("+timesample_forwardshift" to compensate for our fake-reuse of name above)
						interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_old for later evaluation
						// also need to save old stochastic state in textfile in case of timesample, as might vary slightly:
						copy(eta_old.begin(),eta_old.end(),storage.begin()+Nbin*J);	// copy old stohastic state into storage
						J++;
					}
				}	// end timesampling
				copy(rho_old.begin(),rho_old.end(),storage.begin()+Nbin*J);	// copy old deterministic state into storage
				J++;
				interpolate(&rho_old, &rho_old_int, Nbin, interpolationref, interpolationtype);	// interpolate rho_old for later evaluation
			}	// end if-timesample

			// create filename to read and print in saveresults:
			filename = getfilename(dT+timesample_forwardshift, Nbin, micres, Npart_filename, sampling, fsampling, samplesize, wait+fwait+timesample_forwardshift, fwait, type, mode, repeattype, suffix);
			if( withintro )
			{
				cout << "_____" << filename << "_____\n";
			}

			// for timer-output:
			clock_t t3 = clock();			// stop-time of timer
			if( withintro )
			{
				cout << " Waiting done by: " << double(t3-t2)/ CLOCKS_PER_SEC << " sec.\n";
			}
			else if( overallsampling>=10 )	// statesample is active and beyond initialization
			{
				if( modulo(j_overallsampling+1,int(0.1*overallsampling)) < modulo(j_overallsampling,int(0.1*overallsampling)) )
				{
					cout << ".." << (100.*(j_overallsampling+1))/overallsampling << "%.";
					if(j_overallsampling+1 < 0.15*overallsampling)
					{
						clock_t t4 = clock();	// stop-time of timer
						cout << "(" << double(t4-t1)/ CLOCKS_PER_SEC << ")";
					}
				}
				if( j_overallsampling==overallsampling-1 )
				{
					cout << "\n";
				}
			}

			// iterate evolution samplesize-often:
			for( int j=0; j<samplesize; j++ )					// walk along samples (with same initial state Eta_old after waiting time)
			{
				// for control-window-feedback about progress:
				if( samplesize>=10 && withintro )
				{
					if( modulo(j+1,int(0.1*samplesize)) < modulo(j,int(0.1*samplesize)) )
					{
						cout << ".." << (100.*(j+1))/samplesize << "%.";
						if(j+1 < 0.15*samplesize)
						{
							clock_t t4 = clock();				// stop-time of timer
							cout << "(" << double(t4-t3)/ CLOCKS_PER_SEC << ")";
						}
					}
					if( j==samplesize-1 )
					{
						cout << "\n";
					}
				}
				// choose type:
				if( mode == "repeat" || j == 0 )
				{
					Eta_new = Eta_old;							// identify Eta_new with Eta_old, until further evolved
					if( type == "Kawasaki" )
					{
						rate = frate_old;						// remember original jump-rates at Eta_old
					}
					else if( type == "Kawasaki_Ising" )
					{
						S = fS_old; rate = frate_old;			// reset spin-state to original setting to repeat same measurement again
					}
				}
				else if( mode == "continuous" || mode == "continuous2" )
				{
					// do nothing, so evolution continues from previous state Eta_new
				}
				if( type == "ZRP" )
				{
					zero_range_KMC(&Eta_new, &empty, dT, Nmic);	// another version of Eta_new
				}	
				else if( type == "ZRP_multitag" )
				{
					zero_range_KMC_multitag(&Eta_new, &empty, dT, Nmic, &originindex);
					MSD.at(j) = getMSD(&originindex,dT,1.);
					originindex = originindex_memory;			// reset originindex to initial one from Eta_old
				}
				else if( type == "SEP" )
				{
					simple_exclusion_KMC(&Eta_new, dT, Nmic, Npart);	// another version of Eta_new
				}
				else if( type == "Kawasaki" )
				{
					Kawasaki_KMC(&Eta_new, &rate, dT, Nmic);
				}
				else if( type == "Kawasaki_Ising" )
				{
					Kawasaki_Ising_KMC( &S, &rate, 4*Ntot, dT, Nmic, Ny, tilt, b, h, boundarycondition);
					if( repeattype != "timesample" )
					{
						surface.searchstart( &startpt, &S, Ny, Nmic, tilt, &startavoid, &startavoid, lastlength);	// extracts surface
						converttoheight( &Eta_new, surface.getloop(), surface.getnonred_amount(), surface.getloopo(), surface.getnonred_amounto(), &S, Nmic, Ny );	// avoid overhangs to get height instead of surface
						startpt.at(0) = (*surface.getloop_path()).at(0); startpt.at(1) = (*surface.getloopo_path()).at(0);
					}
				}

				if( mode == "repeat" && repeattype == "timesample" )// i.e. if repeattype == timesample
				{
					microtomacro(&Eta_new,&rho_new,Nbin,micres);	// initialize rho_new
					for( int j_ts=0; j_ts<timesample_no; j_ts++ )
					{
						// run process timesample_no often for very short timesample_time to then average over these time-local states to estimate deterministic rho_old:
						if( type == "ZRP" )
						{
							zero_range_KMC(&Eta_new, &empty, timesample_time, Nmic);		// wait intersample time
						}
						else if( type == "ZRP_multitag" )
						{
							cout << " Warning - control: Bad combination: " << type << " and " << repeattype << ".\n";
							return;
						}
						else if( type == "SEP" )
						{
							simple_exclusion_KMC(&Eta_new, timesample_time, Nmic, Npart);	// wait intersample time
						}
						else if( type == "Kawasaki" )
						{
							Kawasaki_KMC(&Eta_new, &rate, timesample_time, Nmic);
						}
						else if( type == "Kawasaki_Ising" )
						{
							Kawasaki_Ising_KMC( &S, &rate, 4*Ntot, timesample_time, Nmic, Ny, tilt, b, h, boundarycondition);
							// get surface:
							surface.searchstart( &startpt, &S, Ny, Nmic, tilt, &startavoid, &startavoid, lastlength);	// extracts surface
							converttoheight( &Eta_new, surface.getloop(), surface.getnonred_amount(), surface.getloopo(), surface.getnonred_amounto(), &S, Nmic, Ny );	// avoid overhangs to get height instead of surface
							startpt.at(0) = (*surface.getloop_path()).at(0); startpt.at(1) = (*surface.getloopo_path()).at(0);
						}
						// update rho_new:
						microtomacro(&Eta_new,&eta_new,Nbin,micres);			// get macroscopical eta_old for the final evaluation in getsigma
						for( int j_rho=0; j_rho<Nbin; j_rho++ )
						{
							rho_new.at(j_rho) = ( rho_new.at(j_rho)*(j_ts+1) + eta_new.at(j_rho) )/(j_ts+2);
						}
						// get eta_new, eta_new_int at half-time to update storage:
						if( j_ts==timesample_no/2 )								// real waiting time done
						{
							copy(eta_new.begin(),eta_new.end(),storage.begin()+Nbin*J);	// copy new stochastic state into storage
							J++;
							interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_old for later evaluation
						}
					}		// end timesampling
					copy(rho_new.begin(),rho_new.end(),storage.begin()+Nbin*J);	// copy new deterministic state into storage
					J++;														// update number of states stored in storage
					interpolate(&rho_new, &rho_new_int, Nbin, interpolationref, interpolationtype);	// interpolate rho_old for later evaluation
				}	//	end if-timesample
				else												// i.e. timesample is not active
				{
					microtomacro(&Eta_new,&eta_new,Nbin,micres);	// create macroscopical final distribution
					// update storage:
					copy(eta_new.begin(),eta_new.end(),storage.begin()+Nbin*J);	// copy new state into storage
					J++;											// update number of states stored in storage
				}

				// write into file if necessary:
				if( J==storagelength )							// time to write into file, as storage is full
				{
					savestatetotext(&storage, dT+timesample_forwardshift, &boundaries, Nbin, micres, Npart_filename, sampling, fsampling, samplesize, wait+fwait+timesample_forwardshift, fwait, type, mode, repeattype, suffix, 0 );	// add to file
					fill(storage.begin(), storage.end(), 0);	// reset storage again (hence filled with zeros, but not reallocated or capacity changed)
					J = 0;										// reset storage-filling-parameter
				}
				
				if( j==samplesize-1 )							// also time to write into file
				{
					vector<double> storage2(storage.begin(),storage.begin()+Nbin*J);	// get new variable for shorter storage
					savestatetotext(&storage2, dT+timesample_forwardshift, &boundaries, Nbin, micres, Npart_filename, sampling, fsampling, samplesize, wait+fwait+timesample_forwardshift, fwait, type, mode, repeattype, suffix, 0 );
					fill(storage2.begin(), storage2.end(), 0); fill(storage.begin(), storage.end(), 0);
					J = 0;																// needed for multiple samplings
				}
				if( ( mode == "repeat" && (repeattype == "keepinitial" || repeattype == "localaverage") ) || mode == "continuous2")
				{
					// update rho_new:
					for( int j_rho=0; j_rho<Nbin; j_rho++ )
					{
						rho_new.at(j_rho) = ( rho_new.at(j_rho)*j + eta_new.at(j_rho) )/(j+1);
					}
				}
				else if( mode == "repeat" && repeattype == "statesample" )
				{
					// update rho_new: (wrt. sampling)
					for( int j_rho=0; j_rho<Nbin; j_rho++ )
					{

						rho_new.at(j_rho) = ( rho_new.at(j_rho)*j_overallsampling + eta_new.at(j_rho) )/(j_overallsampling+1);
					}
				}
				else if( type != "ZRP_multitag" && ( mode == "continuous" || repeattype == "timesample" || repeattype == "purelystochastic" ) )
				{
					if( mode == "continuous" || (mode == "repeat" && repeattype == "purelystochastic") )
					{
						// evaluate states to get sigma:
						interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_new for evaluation
					}
					for( int j_x0=0; j_x0<lx0range; j_x0++ )	// go through all x0 in range
					{
						x0 = x0range[j_x0];						// set current x0
						samplepos = j_x0*totalsample+j_overallsampling*samplesize+j;	// position in samplelists (sigmasample, diametentsample)
						gamma_start = allgamma_im.begin()+Nint*j_x0;
						gammasq_start = allgammasq_im.begin()+Nint*j_x0;
						if( mode == "continuous" )				// first check mode, then repeattype
						{
							Y_old = 0;							// only keep effect of stochastic terms
							Y_new = getY(&eta_old_int, &eta_new_int, gamma_start, Nint, epsilon);
							sigma = getsigma( 0., Y_new - Y_old, &eta_old_int, dt, gammasq_start, Dx );
						}
						else if( repeattype == "timesample" )
						{
							Y_old = getY(&eta_old_int, &rho_old_int, gamma_start, Nint, epsilon);
							Y_new = getY(&eta_new_int, &rho_new_int, gamma_start, Nint, epsilon);
							sigma = getsigma( 0., Y_new - Y_old, &rho_old_int, dt, gammasq_start, Dx );
						}
						else if( repeattype == "purelystochastic" )
						{
							Y_old = 0;							// only keep effect of stochastic terms	(more like Y_deterministic)
							Y_new = getY(&eta_new_int, &eta_old_int, gamma_start, Nint, epsilon);	// more like Y_stochastic
							sigma = getsigma( 0., Y_new - Y_old, &eta_old_int, dt, gammasq_start, Dx );
						}
						diametent = getF((Y_new-Y_old)/sqrt(double(Nmic))) / dt;	// "/sqrt(double(Nmic))" as states in the code are non-normalized (and epsilon = 1 to compensate), but in theory they are (and epsilon = 1/sqrt(Nmic))
						sigmasample.at(samplepos) = sigma;
						diametentsample.at(samplepos) = diametent;
						if( j==0 )								// i.e. only the first time (eta_old and rho_new are fixed anyways)
						{
							eta_a.at(j_x0) = getrho_a(&eta_old_int, Nint, gammasq_start, interpolationref);
							// update eta_a_a:
							eta_a_a.at(j_x0) = ( eta_a_a.at(j_x0)*j_overallsampling + eta_a.at(j_x0) )/(j_overallsampling+1);
						}
						if( j==samplesize-1  )					// write very final state as rho_a
						{
							rho_a.at(j_x0) = getrho_a(&eta_new_int, Nint, gammasq_start, interpolationref);
							// update rho_a_a:
							rho_a_a.at(j_x0) = ( rho_a_a.at(j_x0)*j_overallsampling + rho_a.at(j_x0) )/(j_overallsampling+1);
							// get del_t rho (i.e. lhs of gradient-flow):
							drhodT.at(j_x0) = getY( &rho_new_int, &eta_old_int, gamma_start, Nint, pow(Nmic*dT/pow(Nmic,2.0),2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are
						}
					}	// end x0-range
					if( mode == "continuous" )
					{
						copy(eta_new_int.begin(), eta_new_int.end(), eta_old_int.begin());	// update: future old state is current new state
					}
				}	// end mode-specific evaluations
			}	// end of creating sample		

			// read file again, to find sigma:
			if( type!="ZRP_multitag" && ( ( mode == "repeat" && (repeattype == "keepinitial" || repeattype == "localaverage" ) ) || mode == "continuous2" ) )
			{
				if( repeattype == "localaverage" )
				{
					// get softened rho_old for evaluation of Y_old:
					microtomacro_double(&rho_new,&rho_new_soft,Nsoft,softener);
					interpolate(&rho_new_soft, &rho_new_int, Nsoft, interpolationref*softener,"const");
					for( int j_soft=0; j_soft<Nint; j_soft++ )
					{	
						rho_new_int.at(j_soft) = rho_new_int.at(j_soft)*softener;
					}
				}
				else if( repeattype == "keepinitial" )
				{
					// interpolate rho_new: (rho_old==eta_old)
					interpolate(&rho_new, &rho_new_int, Nbin, interpolationref, interpolationtype);
				}

				clock_t t5 = clock();			// stop-time of timer
				string str;						// buffer for boring string-readings
				double value = -1;				// buffer for reading entries in state-vectors
				j_sigmasample = 0;				// reset j_sigmasample (index within samplesize), such that it only runs within the sampling-reading
				ifstream textfile( filename );	// open textfile for reading

				if( textfile.is_open() )
				{
					// read headlines:
					textfile >> str >> dT >> str >> boundaries.at(0) >> boundaries.at(1) >> str >> Nbin >> str >> micres >> str >> Npart >> str >> samplesize >> str >> fsampling >> str >> sampling >> str >> wait >> str >> fwait >> str >> mode >> str >> repeattype >> str >> suffix;
					if( j_sampling == 0 )
					{
						cout << "dT = " << dT << ", boundaries = " << boundaries.at(0) << ".." << boundaries.at(1) << ", Nbin = " << Nbin << ", micres = "<< micres << ", Npart = " << Npart << ", sampling = " << sampling <<", fsampling = " << fsampling << ", samplesize = " << samplesize << ", wait = " << wait << ", fwait = " << fwait << ", mode = " << mode << ", repeattype = " << repeattype << ", suffix = " << suffix << "\n";
						cout << " refinement = " << refinement << "\n";
						cout << " interpolation = " << interpolationtype << " (" << interpolationref << ")\n";
					}
					// read Eta_old:
					while( textfile >> value )						// always read until end of storagelength in units of Nmic (i.e. eta_old) (value is int)
					{
						// don't do anything
					}	// end of storagelength-while-loop
					// ignore x-delimiters:
					textfile.clear();								// clear all flags of textfile, so no bitfail occurs
					textfile.ignore();								// ignore 'x'
					// read states:
					while(!textfile.eof())							// check if file is still not at end
					{
						J = 0;										// reset index within eta_new
						while( textfile >> value )					// always read until end of storagelength in units of Nmic (i.e. one eta_new)
						{
							eta_new.at(J) = value;
							J++;									// update index within eta_new
							if( J==Nbin )							// i.e. have read one complete eta_new (note, that J is already updated for next measurement)
							{
								// interpolate read state:
								interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);
								// go through space and determine diffusivities:
								for( int j_x0=0; j_x0<lx0range; j_x0++ )	// go through all x0 in range
								{
									x0 = x0range[j_x0];				// set current x0
									samplepos = j_x0*totalsample+j_overallsampling*samplesize+j_sigmasample;	// position in samplelists (sigmasample, diametentsample)
									gamma_start = allgamma_im.begin()+Nint*j_x0;
									gammasq_start = allgammasq_im.begin()+Nint*j_x0;
									if( repeattype == "localaverage" )
									{
										Y_old = getY(&rho_old_int, &eta_old_int, gamma_start, Nint, epsilon);
										Y_new = getY(&rho_new_int, &eta_new_int, gamma_start, Nint, epsilon);
										sigma = getsigma( 0, Y_new-Y_old, &rho_old_int, dt, gammasq_start, Dx );
									}
									else if( repeattype == "keepinitial" )
									{
										Y_old = 0;						// keep initial state constant and identify with dterministic rho_old
										Y_new = getY(&rho_new_int, &eta_new_int, gamma_start, Nint, epsilon);
										sigma = getsigma( 0, Y_new-Y_old, &eta_old_int, dt, gammasq_start, Dx );
									}
									else
									{
										cout << " Warning - control: Unknown repeattype: " << repeattype << "\n";
										return;
									}
									diametent = getF((Y_new-Y_old)/sqrt(double(Nmic))) / dt;	// "/sqrt(double(Nmic))" as states in the code are non-normalized (and epsilon = 1 to compensate), but in theory they are (and epsilon = 1/sqrt(Nmic))
									sigmasample.at(samplepos) = sigma;
									diametentsample.at(samplepos) = diametent;
									if( j_sigmasample==0 )				// i.e. only the first time (eta_old and rho_new are fixed anyways)
									{
										// update eta_a, eta_a_a:
										eta_a.at(j_x0) = getrho_a(&eta_old_int, Nint, gammasq_start, interpolationref);
										eta_a_a.at(j_x0) = ( eta_a_a.at(j_x0)*j_overallsampling + eta_a.at(j_x0) )/(j_overallsampling+1);
										// update rho_a, rho_a_a:
										rho_a.at(j_x0) = getrho_a(&eta_new_int, Nint, gammasq_start, interpolationref);
										rho_a_a.at(j_x0) = ( rho_a_a.at(j_x0)*j_overallsampling + rho_a.at(j_x0) )/(j_overallsampling+1);
										// get del_t rho (i.e. lhs of gradient-flow):
										drhodT.at(j_x0) = getY( &rho_new_int, &eta_old_int, gamma_start, Nint, pow(Nmic*dT/pow(Nmic,2.0),2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are
									}
								}
								j_sigmasample++;				// update sigma-sample-index for reading next eta_new
								// reset J to refill eta_new with new one:
								J = 0;
								// for control-window-feedback about progress:
								if( samplesize>=10 )
								{
									if( modulo(j_sigmasample,int(0.1*samplesize)) < modulo(j_sigmasample-1,int(0.1*samplesize)) )
									{
										cout << ".." << (100.*j_sigmasample)/samplesize << "%.";
										if(j_sigmasample+1 < 0.15*samplesize)
										{
											clock_t t6 = clock();	// stop-time of timer
											cout << "(" << double(t6-t5)/ CLOCKS_PER_SEC << ")";
										}
									}
									if( j_sigmasample==samplesize )
									{
										cout << "\n";
									}
								}
							}	// end processing eta_new
						}	// end of storagelength-while-loop
						// ignore x-delimiters:
						textfile.clear();					// clear all flags of textfile, so no bitfail occurs
						textfile.ignore();					// ignore 'x'
					}	// end of "end of file"-while-loop
				}	// end of file-open-if
				else										// sth went wrong
				{
					cout << "  Warning - control: Could not open the textfile to read state!\n(" << filename << ")\n";
				}
			}	// end if-repeat
			clock_t t7 = clock();							// start-time of timer
			if( withintro )
			{
				cout << " Total time-consumption of: " << double(t7-t2)/ CLOCKS_PER_SEC << " sec.\n\n";
			}

			// update overallsampling:
			j_overallsampling++;							// j_fsampling+j_sampling
		}	// end fsampling-loop
	}	// end sampling-loop

	// do repeattype == "statesample" after taking all the samplings:
	if( type!="ZRP_multitag" && (mode == "repeat" && repeattype == "statesample") )
	{
		// interpolate for post-processable states:
		interpolate(&rho_old, &rho_old_int, Nbin, interpolationref, interpolationtype);
		interpolate(&rho_new, &rho_new_int, Nbin, interpolationref, interpolationtype);

		clock_t t5 = clock();								// stop-time of timer
		string str;											// buffer for boring string-readings
		double value = -1;									// buffer for reading entries in state-vectors
		j_sigmasample = 0;									// reset j_sigmasample (index within sampling and fsampling), such that it only runs within the sampling-reading
		ifstream textfile3( filename );						// open textfile for reading
		
		if( textfile3.is_open() )
		{
			// read headlines:
			textfile3 >> str >> dT >> str >> boundaries.at(0) >> boundaries.at(1) >> str >> Nbin >> str >> micres >> str >> Npart >> str >> samplesize >> str >> fsampling >> str >> sampling >> str >> wait >> str >> fwait >> str >> mode >> str >> repeattype >> str >> suffix;
			cout << "dT = " << dT << ", boundaries = " << boundaries.at(0) << ".." << boundaries.at(1) << ", Nbin = " << Nbin << ", micres = "<< micres << ", Npart = " << Npart << ", sampling = " << sampling <<", fsampling = " << fsampling << ", samplesize = " << samplesize << ", wait = " << wait << ", fwait = " << fwait << ", mode = " << mode << ", repeattype = " << repeattype << ", suffix = " << suffix << "\n";
			cout << " refinement = " << refinement << "\n";
			cout << " interpolation = " << interpolationtype << " (" << interpolationref << ")\n";

			// read states:
			while(!textfile3.eof())							// check if file is still not at end
			{
				J = 0;										// reset index within Eta_old
				// read Eta_old:
				while( textfile3 >> value )					// always read until end of storagelength in units of Nmic (i.e. eta_old) (value is int)
				{
					Eta_old.at(J) = value;
					J++;
				}	// end of storagelength-while-loop
				// interpolate read state:
				microtomacro(&Eta_old,&eta_old,Nbin,micres);
				interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);
				// ignore x-delimiters:
				textfile3.clear();							// clear all flags of textfile, so no bitfail occurs
				textfile3.ignore();							// ignore 'x'
				if( textfile3.eof() )						// last entry survives sometimes
				{
					break;
				}
				J = 0;										// reset index within eta_new
				while( textfile3 >> value )					// always read until end of storagelength in units of Nmic (i.e. one eta_new)
				{
					eta_new.at(J) = value;
					J++;									// update index within eta_new
					if( J==Nbin )							// currently not needed as samplesize == 1 for statesample
					{
						// interpolate read state:
						interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);
						// go through space and determine diffusivities:
						for( int j_x0=0; j_x0<lx0range; j_x0++ )		// go through all x0 in range
						{
							x0 = x0range[j_x0];				// set current x0
							samplepos = j_x0*totalsample+j_sigmasample;	// position in samplelists (sigmasample, diametentsample)
							// evaluate sigma:
							gamma_start = allgamma_im.begin()+Nint*j_x0;
							gammasq_start = allgammasq_im.begin()+Nint*j_x0;
							Y_old = getY(&rho_old_int, &eta_old_int, gamma_start, Nint, epsilon);
							Y_new = getY(&rho_new_int, &eta_new_int, gamma_start, Nint, epsilon);
							sigma = getsigma( 0, Y_new-Y_old, &rho_old_int, dt, gammasq_start, Dx );
							diametent = getF((Y_new-Y_old)/sqrt(double(Nmic))) / dt;	// "/sqrt(double(Nmic))" as states in the code are non-normalized (and epsilon = 1 to compensate), but in theory they are (and epsilon = 1/sqrt(Nmic))
							sigmasample.at(samplepos) = sigma;
							diametentsample.at(samplepos) = diametent;
							if( j_sigmasample==0 )			// i.e. only the first time (rho_old and rho_new are fixed anyways)
							{
								// update eta_a, eta_a_a:
								eta_a.at(j_x0) = getrho_a(&rho_old_int, Nint, gammasq_start, interpolationref);
								eta_a_a.at(j_x0) = eta_a.at(j_x0);		// identical for statesample
								// update rho_a, rho_a_a:
								rho_a.at(j_x0) = getrho_a(&rho_new_int, Nint, gammasq_start, interpolationref);
								rho_a_a.at(j_x0) = rho_a.at(j_x0);		// identical for statesample
								// get del_t rho (i.e. lhs of gradient-flow):
								drhodT.at(j_x0) = getY( &rho_new_int, &rho_old_int, gamma_start, Nint, pow(Nmic*dT/pow(Nmic,2.0),2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are
							}
						}	// end xrange-evaluation
						j_sigmasample++;					// update sigma-sample-index for reading next eta_new
						// reset J to refill eta_new with new one:
						J = 0;
						// for control-window-feedback about progress:
						if( overallsampling>=10 )
						{
							if( modulo(j_sigmasample,int(0.1*overallsampling)) < modulo(j_sigmasample-1,int(0.1*overallsampling)) )
							{
								cout << ".." << (100.*j_sigmasample)/overallsampling << "%.";
								if(j_sigmasample+1 < 0.15*overallsampling)
								{
									clock_t t6 = clock();	// stop-time of timer
									cout << "(" << double(t6-t5)/ CLOCKS_PER_SEC << ")";
								}
							}
							if( j_sigmasample==overallsampling )
							{
								cout << "\n";
							}
						}
					}	// end processing eta_new
				}	// end of storagelength-while-loop
				// ignore x-delimiters:
				textfile3.clear();							// clear all flags of textfile, so no bitfail occurs
				textfile3.ignore();							// ignore 'x'
			}	// end of "end of file"-while-loop
		}	// end of file-open-if
		else												// sth went wrong
		{
			cout << "  Warning - control: Could not open the textfile to read state!\n(" << filename << ")\n";
		}
	}	// end if repeattype == "statesample"
	
	// for output in controlwindow:
	if( type != "ZRP_multitag" )
	{
		for( int j_x0=0; j_x0<lx0range; j_x0++ )
		{
			samplepos = j_x0*totalsample;
			nextsamplepos = (j_x0+1)*totalsample;
			average.at(j_x0) = accumulate(samplepos+sigmasample.begin(), nextsamplepos+sigmasample.begin(), 0.)/totalsample;
			stdderr.at(j_x0) = sqrt( ( inner_product(samplepos+sigmasample.begin(), nextsamplepos+sigmasample.begin(), samplepos+sigmasample.begin(), 0.)/totalsample - pow(average.at(j_x0),2.) )/(totalsample-1) );
			cout << " sigma(rho("<< x0range[j_x0] <<")) = " << "sigma("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << average.at(j_x0) << "  +-  " << stdderr.at(j_x0) << "\n";
			
			cout << " <gamma,drho/dT> = " << drhodT[j_x0] << "\n";
			avdiametent.at(j_x0) = accumulate(samplepos+diametentsample.begin(), nextsamplepos+diametentsample.begin(), 0.)/totalsample;
			stderr_diametent.at(j_x0) = sqrt( ( inner_product(samplepos+diametentsample.begin(), nextsamplepos+diametentsample.begin(), samplepos+diametentsample.begin(), 0.)/totalsample - pow(avdiametent.at(j_x0),2.) )/(totalsample-1) );
			cout << " diametent(rho("<< x0range[j_x0] <<")) = " << "diametent("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << avdiametent.at(j_x0) << "  +-  " << stderr_diametent.at(j_x0) << "\n";
		}	// end x0-loop
	}	// end of non-multitag
	else	// i.e. if multitag
	{
		average.push_back( accumulate(MSD.begin(),MSD.end(),0.)/totalsample );
		stdderr.push_back( sqrt( ( inner_product(MSD.begin(), MSD.end(), MSD.begin(), 0.)/totalsample - pow(average.at(0),2.) )/(totalsample-1) ) );
		cout << " MSD = " << average.at(0) << " +- " << stdderr.at(0) << "\n";
	}
	
	// to save the calculated data externally:
	saveresultstotext( x0range, lx0range, refinement, &eta_a, &rho_a, &average, &stdderr, &drhodT, &avdiametent, &stderr_diametent, filename, t1, storagelength, filtertype, filterparameters_adr, interpolationtype, interpolationref, samplingmode, sampling, fsampling );
	
	// for timer-output:
	clock_t t8 = clock();	// stop-time of timer
	cout << " end of control-function after " << double(t8-t1)/ CLOCKS_PER_SEC << " sec.\n\n";

	return;
}