/*
	In this function the main physical and numerical parameters are set and the individual subfunctions are called
*/

#include <iostream>		// needed for output in control-window
#include <ctime>		// needed for timer
#include <string>		// needed for strings/names
#include <sstream>		// needed for ostringstream
#include <vector>		// needed for vectors/vector-operations

using namespace std;

#include "control.h"
#include "translate.h"

int main()
{
	// for timer:
	clock_t t1 = clock();					// start-time of timer
	// initialize the physcial parameter:
	static int Nbinrange[] = {5000};		// Nbin: number of (macroscopical) sites for spatial discretization of macroscopic state
	static int micresrange[] = {1};			// micres: number of sites per macroscopical site (i.e. total number of sites = Nbin*micres)
	static int Npartrange[] = {4750};		// Npart: proportional to total number of particles in the system
	static int samplesizerange[] = {1};		// samplesize: number of samples that contribute to one diffusivity-measurement (both for determining the deterministic state and the deviation from it)
	static double dTrange[] = {0.001};		// dT: microscopic measurement time
	static double waitrange[] = {100};		// wait: microscopic initial waiting time after the initialization and the start of the measurement
	static double fwaitrange[] = {0.1};		// preinitial waiting time to get fluctuations right
	static double x0range[] = {0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9};// x0: concentration-point of the filter-function gamma
	string samplingmode = "combined";		// "separate" for separate output for each sampling, "combined" for one output for all simplings combined
	static int samplingrange[] = {50};		// sampling: number of samples with a new evolution during the entire preinitial waiting time
	static int fsamplingrange[] = {2000};	// fsampling: number of samples with a new evolution during the preinitial fwait period of time (not recommended to be combined with multiple separate samplings, as then different measurements get written in same file)

	// initialize in-loop-parameters:
	string initializationtype = "wait";		// "nowait" only for analytic equilibrium-initialization with ZRP-processes
	string profile = "sin";			    	// profile of initial density ("flat" for constant, "sin" for half-sin, etc.)
	string mode = "repeat";					// "repeat" to start from same initial condition again and again, "continuous" for one long measurement with measurement outtakes in between
	string repeattype = "statesample";		// for specification of how "deterministic states" are found
	string type = "SEP";					// type of stochastic process: "ZRP" = zero-range-process, "SEP" = simple-exclusion-process, "Kawasaki" = pseudo-Kawasaki, "Kawasaki_Ising" = Kawasaki
	string name = "";						// name-suffix for external textfile (no effect on the )
	int storagelength = 2000;				// amount of iterations before states get written in external file
	stringstream filename;					// initialize the stringstream for the name
	const int lx0range = int( sizeof(x0range)/sizeof(x0range[0]) );	// amount of elements in x0range
	string filtertype = "sw";				// type for filter-function gamma
	static vector<double> filterparameters;	// parameters in filter-function gamma; first argument is "a0", second is "a1" etc.
	filterparameters.push_back(1.);			// initialize a0
	filterparameters.push_back(160.);		// initialize a1
	filterparameters.push_back(2.);			// initialize a2
	string interpolationtype = "const";		// interpolation-type
	int interpolationref = 1;				// refinement for interpolation
	double refinement = 1;					// factor, how much finer the resolution shall be for determining the derivatives (nabla gamma, dF^2/dx^2)/ for dx -> Dx

	for( int jsampling=0; jsampling<sizeof(samplingrange)/sizeof(samplingrange[0]); jsampling++)
	{
		int sampling = samplingrange[jsampling];
		for( int jfsampling=0; jfsampling<sizeof(fsamplingrange)/sizeof(fsamplingrange[0]); jfsampling++ )
		{
			int fsampling = fsamplingrange[jfsampling];
			for( int jsamplesize=0; jsamplesize<sizeof(samplesizerange)/sizeof(samplesizerange[0]); jsamplesize++ )
			{
				int samplesize = samplesizerange[jsamplesize];
				for( int jdT=0; jdT<sizeof(dTrange)/sizeof(dTrange[0]); jdT++ )
				{
					double dT = dTrange[jdT];
					for( int jwait=0; jwait<sizeof(waitrange)/sizeof(waitrange[0]); jwait++ )
					{
						double wait = waitrange[jwait];
						for( int jfwait=0; jfwait<sizeof(fwaitrange)/sizeof(fwaitrange[0]); jfwait++ )
						{
							double fwait = fwaitrange[jfwait];
							for( int jmicres=0; jmicres<sizeof(micresrange)/sizeof(micresrange[0]); jmicres++ )
							{
								int micres = micresrange[jmicres];
								for( int jNbin=0; jNbin<sizeof(Nbinrange)/sizeof(Nbinrange[0]); jNbin++ )
								{
									int Nbin = Nbinrange[jNbin];
									for( int jNpart=0; jNpart<sizeof(Npartrange)/sizeof(Npartrange[0]); jNpart++ )
									{
										int Npart = Npartrange[jNpart];
										if( samplingmode == "separate" )
										{
											for( int jsampling=0; jsampling<sampling; jsampling++ )
											{
												filename.str("");		// clear earlier content of the name-stringstream
												filename << filtertype << "_" << interpolationtype << "(" << interpolationref << ")" << "_" << profile << "_" << name << "_" << jsampling+1;
												control( dT, Nbin, micres, Npart, wait, fwait, x0range, lx0range, refinement, samplesize, storagelength, filename.str(), samplingmode, 1, fsampling, type, mode, repeattype, profile, filtertype, &filterparameters, interpolationtype, interpolationref, initializationtype );	// run actual programme
											}	// end jsampling
										}
										else if( samplingmode == "combined" )
										{
											control( dT, Nbin, micres, Npart, wait, fwait, x0range, lx0range, refinement, samplesize, storagelength, name, samplingmode, sampling, fsampling, type, mode, repeattype, profile, filtertype, &filterparameters, interpolationtype, interpolationref, initializationtype );	// run actual programme
										}
										else
										{
											cout << "Warning - main: Wrong samplingmode: " << samplingmode << "\n";
										}
									}	// end jNpart
								}	// end jNbin
							}	// end jmicres
						}	// end fwait
					}	// end jwait
				}	// end jdT
			}	// end jsamplesize
		}	// end jfsampling
	}	// end jsampling

	// output in control-window:
	// for timer and final message:
	clock_t t2 = clock();	// stop-time of timer
	cout << "Finddiffusivity total CPU-time-consumption:\n";
	cout << "  " << double(t2-t1)/ CLOCKS_PER_SEC << " sec.\n";

	system("pause");		// for keeping command-window alive

	return 0;
}
