/*
	symmetric simple-exclusion-process in 1D with absorbing or periodic boundary-conditions
*/

#include <iostream>		// needed for output in control-window
#include <vector>		// needed for vectors/vector-operations
#include <random>		// needed for random-number-generator
#include <math.h>		// needed for log

using namespace std;

#include "modulo.h"		// for periodic boundary-conditions

void simple_exclusion_KMC( vector<int>* Eta_new_adr, double dT, int sitenumber, int Npart )
{
	double time = 0;					// initial physical time
	
	// initialize random distribution:
	random_device rd;
	mt19937_64 seed(rd());
	uniform_real_distribution<double> distribution(0.0,1.0);
	double randno1 = 0, randno2 = 0;	// initializes random number between 0 and 1
	int direction = 0;					// initialize variable, that decides if to jump right or left

	// initialize loop-variables:
	int next = -1, jumpx = -1, jumptox = -1;
	bool cond = 1;						// condition to set periodic or absorbing boundary-conditions, see below
	vector<int> idtopos(Npart);			// translates from the particle-no to its position
	int counter = 0;
	for( int j=0; j<sitenumber; j++ )
	{
		if( (*Eta_new_adr).at(j) == 1 )	// Eta_new has only entries 0 or 1
		{
			idtopos.at(counter) = j;
			counter++;
		}
	}

	while( time <= dT )												// run until physical time "time" exceeds the preset measurement time "dT"
	{
		// create random number:
		randno1 = distribution(seed);								// to decide, which particle jumps
		if( randno1 > 0.5 )
		{
			direction = 1;											// particle jumps right (if possible)
			randno1 += -0.5;										// map randno1 to only the half-space, i.e. uniformly distributed in [0,0.5]
		}
		else
		{
			direction = 0;											// particle jumps left
		}
		randno2 = distribution(seed);								// to decide, how long it took to jump
		// decide, which particle is the next one to jump and carry out jump:
		next = int(floor( 2.*randno1 * Npart ));					// particle-no of next particle to jump
		jumpx = idtopos.at(next);									// position, from where the next particle jumps
		time += -log(randno2)/Npart;								// need new random number to avoid correlation between waiting time and location
		if( time<=dT )												// don't jump, if new time is beyond terminal time
		{
			if( direction > 0.5 )									// jump to the right
			{
				jumptox = modulo(jumpx+1,sitenumber);				// take into account periodic boundary conditions
			}
			else													// jump to the left
			{
				jumptox = modulo(jumpx-1,sitenumber);
			}
			cond = ( 1 || jumptox-jumpx==1 || jumptox-jumpx==-1 );	// set first condition to "1" for periodic boundary-conditions or to "0" for absorbing boundary-conditions
			if( cond )												// boundary-condition is meet, i.e. particle is allowed to jump somewhere inside the xrange
			{
				if( (*Eta_new_adr).at(jumptox) == 0 )				// check, if neighbouring site is unoccupied
				{
					(*Eta_new_adr).at(jumpx) += -1;					// particle jumps away from jumpx
					(*Eta_new_adr).at(jumptox) += 1;				// particle jumped to "jumptox"
					idtopos.at(next) = jumptox;						// update translation-list
				}	// end check, if neighbouring place is not occupied
			}
			else													// particle jumped out of the intervall
			{
				(*Eta_new_adr).at(jumpx) += -1;						// particle jumps away from jumpx
				idtopos.erase(idtopos.begin()+next);				// update translation-list
				Npart--;											// one particle less inside the intervall
			}
		}	// end time not beyond dT
	}

	return;
}