/*
	zero-range-process in 1D with absorbing or periodic boundary-conditions

	in case no particle-tagging is on, Eta_old has to be set as an empty vector
	otherwise Eta_old is the distribution of all particles, whereas eta_old is the distribution of the tagged particles
	note: here tagged particle simply means a second species of particles, that are distinguished
*/

#include <vector>		// needed for vectors/vector-operations
#include <random>		// needed for random-number-generator
#include <math.h>		// needed for log

using namespace std;

#include "binsearch.h"
#include "modulo.h"		// for periodic boundary-conditions
#include "g.h"			// jump-rate per site

void zero_range_KMC( vector<int>* eta_old_adr, vector<int>* Eta_old_adr, double dT, int sitenumber )
{
	double time = 0;					// initial physical time
	vector<double> cumsum(sitenumber);	// cumulated sum of jump-rates of the sites in order of eta_old;
	double lastcumsum = 0;				// last element of cumsum (i.e. sum over all jump-rates)
	int taggingon = 1;					// assume tagging unless...
	if( (*Eta_old_adr).empty() )		// ...Eta_old is empty
	{
		Eta_old_adr = eta_old_adr;		// identify both eta_olds
		taggingon = 0;					// no tagging
	}

	// initialize random distribution:
	random_device rd;
	mt19937_64 seed(rd());
	uniform_real_distribution<double> distribution(0.0,1.0);
	double randno1 = 0, randno2 = 0;	// initializes random number between 0 and 1
	double randno3 = 0;
	int direction = 0;					// initialize variable, that decides if to jump right or left

	// initialize loop-variables:
	int jumpx = -1, jumptox = -1;
	bool cond = 1;						// condition to set periodic or absorbing boundary-conditions, see below
	double searchat = 0.0;

	while( time <= dT )													// run until physical time "time" exceeds the preset measurement time "dT"
	{
		// compute cumulated sum:
		cumsum.at(0) = g( (*Eta_old_adr).at(0) );
		for( int j=1; j<sitenumber; j++ )								// walk along all sites
		{
			cumsum.at(j) = cumsum.at(j-1) + g( (*Eta_old_adr).at(j) );	// g is jump-rate per site with as many particles as in the argument
		}
		lastcumsum = cumsum.at(sitenumber-1);							// last entry in cumsum
		// create random number:
		randno1 = distribution(seed);
		if( randno1 > 0.5 )
		{
			direction = 1;												// particle jumps right (if possible)
			randno1 += -0.5;											// map randno1 to only the half-space, i.e. uniformly distributed in [0,0.5]
		}
		else
		{
			direction = -1;												// particle jumps left
		}
		randno2 = distribution(seed);
		// carry out jump:
		searchat = 2*randno1*lastcumsum;
		jumpx = binsearch(&cumsum, searchat, max(0.,floor(2*randno1*sitenumber-sqrt(2*randno1*sitenumber))), min(sitenumber-1.0,ceil(2*randno1*sitenumber+sqrt(2*randno1*sitenumber))));	// position, where particle jumps from (to left or right)
		time += -log(randno2)/lastcumsum;								// need new random number to avoid correlation between waiting time and location
		if( time<=dT )													// don't jump, if new time is beyond terminal time
		{
			jumptox = modulo(jumpx+direction,sitenumber);
			(*Eta_old_adr).at(jumpx) += -1;								// particle jumps away from jumpx
			cond = ( 1 || jumptox-jumpx==1 || jumptox-jumpx==-1 );		// set first condition to "1" for periodic boundary-conditions or to "0" for absorbing boundary-conditions
			if( cond )													// boundary-condition is meet, i.e. particle is allowed to jump somewhere inside the xrange
			{
				(*Eta_old_adr).at(jumptox) += 1;						// particle jumped to "jumptox"
			}
			// if particle-tagging is on, also update distribution eta_old of tagged particles:
			if( taggingon )
			{
				randno3 = distribution(seed);							// create new random number
				if( randno3 < (*eta_old_adr).at(jumpx)/((*Eta_old_adr).at(jumpx)+1) )	// a tagged particle jumped
				{
					(*eta_old_adr).at(jumpx) += -1;						// tagged particle jumped away from jumpx
					if( cond )
					{
						(*eta_old_adr).at(jumptox) += 1;				// tagged particle arrived inside the xrange
					}
				}
			}	// end taggingon
		}	// end time not beyond dT
	}	// end time-evolution

	return;
}