/*
	Computes Gram/Schmidt transformation for basis/vectors and outputs text-file with prefactors
*/


#include <iostream>		// needed for output in control-window
#include <sstream>		// needed for ostringstream
#include <fstream>		// needed for output to external files
#include <ctime>		// needed for timer
#include <string>		// needed for strings/names
#include <vector>		// needed for vectors/vector-operations
#include <numeric>		// needed for inner_product
#include <algorithm>	// needed for transform (in test)

using namespace std;

void gramschmidt( vector<double>::iterator gamma_im_start, int lx0range, int sitenumber, double dx, string filtertype, vector<double>* filterparameters_adr )
{
	double f,h, norm;									// projection of full- or half-overlapping basis-vectors
	vector<double> coef(lx0range*lx0range);				// coefficients of new orthonormal basis wrt old basis: <new1|old1>, <new1|old2>, ...
	f = inner_product(gamma_im_start, gamma_im_start+sitenumber, gamma_im_start, 0.) *dx;
	h = inner_product(gamma_im_start, gamma_im_start+sitenumber, gamma_im_start+sitenumber, 0.) *dx;

	for( int j_new=0; j_new<lx0range; j_new++ )			// go through new basis
	{
		// get direction:
		for( int j_old=0; j_old<j_new+1; j_old++ )		// go through old basis
		{
			if( j_old<j_new )
			{
				coef.at(j_new*lx0range+j_old) = -h*coef.at((j_new-1)*lx0range+j_new-1)*coef.at((j_new-1)*lx0range+j_old);
			}
			else	// i.e. for j_old==j_new
			{
				coef.at(j_new*lx0range+j_old) = 1;
			}
		}
		// get normalized length:
		norm =          f*inner_product(coef.begin()+j_new*lx0range,coef.begin()+(j_new+1)*lx0range,  coef.begin()+j_new*lx0range,  0.);	// pure parts
		norm = norm + 2*h*inner_product(coef.begin()+j_new*lx0range,coef.begin()+(j_new+1)*lx0range-1,coef.begin()+j_new*lx0range+1,0.);	// mixed parts
		norm = sqrt(norm);
		for( int j_old=0; j_old<j_new+1; j_old++ )		// go through old basis
		{
			coef.at(j_new*lx0range+j_old) = coef.at(j_new*lx0range+j_old)/norm;
		}
	}

	// write into external file:
	stringstream filename;								// name of file, where coefficients will be saved in
	filename << "Gram-Schmidt-coefficients for " << filtertype << " (";
	for( int j_par=0; j_par<(*filterparameters_adr).size(); j_par++ )
	{
		filename << (*filterparameters_adr).at(j_par) << " ";
	}
	filename << ").txt";
	ofstream textfile( filename.str() );				// opens/ creates file "filename", referred to as textfile here
	if( textfile.is_open() )							// just check to be save...
	{
		textfile.precision(10);							// set precision up to ten digits (unless limited by floatfield)
		for( int j_new=0; j_new<lx0range; j_new++ )		// go through new basis
		{
			for( int j_old=0; j_old<lx0range; j_old++ )	// go through old basis
			{
				textfile << coef.at(j_new*lx0range+j_old) << "\t";
			}
			textfile << "\n";
		}
		textfile.close();								// close textfile again
	}
	else												// sth went wrong
	{
		cout << "  Warning - gramschmidt: Could not open a textfile to save state!\n(" << filename.str() << ")\n";
	}


	// test:
	vector<double> newbasis(sitenumber*lx0range);		// new set of basisvectors, one after another
	vector<double> oldproj(sitenumber);
	for( int j_new=0; j_new<lx0range; j_new++ )
	{
		for( int j_old=0; j_old<lx0range; j_old++ )
		{
			transform(gamma_im_start+j_old*sitenumber,   gamma_im_start+(j_old+1)*sitenumber,   oldproj.begin(), bind1st(multiplies<double>(),      coef.at(j_new*lx0range+j_old)));
			transform(newbasis.begin()+j_new*sitenumber, newbasis.begin()+(j_new+1)*sitenumber, oldproj.begin(), newbasis.begin()+j_new*sitenumber, plus<double>() );
		}
	}
	ofstream textfile2( filename.str(), ofstream::app );// opens/ creates file "filename" to append, referred to as textfile2 here
	textfile2 << "\n\n";
	if( textfile2.is_open() )							// just check to be save...
	{
		textfile2.precision(10);						// set precision up to ten digits (unless limited by floatfield)
		// projections:
		for( int j1=0; j1<lx0range; j1++ )
		{
			for( int j2=0; j2<j1+1; j2++ )
			{
				textfile2 << dx*inner_product(newbasis.begin()+j1*sitenumber, newbasis.begin()+(j1+1)*sitenumber, newbasis.begin()+j2*sitenumber, 0. ) << "\t";
			}
			textfile2 << "\n";
		}
		textfile2 << "\n\n\n";
		// new basis/vectors:
		for( int j_new=0; j_new<lx0range; j_new++ )
		{
			for( int x=0; x<sitenumber; x++ )
			{
				textfile2 << newbasis.at(j_new*sitenumber+x) << "\t";
			}
			textfile2 << "\n\n";
		}
		textfile2.close();								// close textfile again
	}
	else												// sth went wrong
	{
		cout << "  Warning - gramschmidt: Could not open a textfile to save state!\n(" << filename.str() << ")\n";
	}

	return;
}