/*
	reads files as written by savestatetotext.cpp and evaluates them analogous to control.cpp from finddiffusivity
*/

#include <iostream>		// needed for output in control-window
#include <ctime>		// needed for timer
#include <string>		// needed for strings/names
#include <vector>		// needed for vectors/vector-operations
#include <numeric>		// needed for accumulate or innerproduct
#include <algorithm>	// needed for copy, max
#include <fstream>		// needed for output to/input from external files
#include <sstream>		// needed for string-streaming
#include <math.h>		// needed for sqrt and pow

using namespace std;

#include "microtomacro.h"
#include "microtomacro_double.h"
#include "modulo.h"
#include "getxrange.h"
#include "gamma.h"
#include "getY.h"
#include "getsigma.h"
#include "getrho_a.h"
#include "getF.h"
#include "saveresultstotext.h"
#include "interpolate.h"

void readagainC( vector<string>* namelist, int readsampling, string samplingmode, string filtertype, vector<double>* filterparameters_adr, double x0range[], const int lx0range, double refinement, bool diffscaling, double epsilon, string interpolationtype, int interpolationref  )
{
	// for timer:
	clock_t t1 = clock();								// start-time of timer

	// check to be safe:
	if( samplingmode != "separate" && samplingmode != "combined" )
	{
		cout << "Warning - readagainC: Unknown samplingmode: " << samplingmode << "\n";
		return;
	}

	// initialize output parameters:
	vector<double> average(lx0range), stdderr(lx0range);// list of results for sigma and its error at each point x0
	vector<double> avdiametent(lx0range), stderr_diametent(lx0range);	// list of result for diagonal metric entries and its standard deviation

	// initialize reading parameters:
	double dT, dt, wait, fwait;							// microscopic and macroscopic measurement and waiting time
	int Nbin, Nint, micres, Npart, samplesize, fsampling, sampling, storagelength = 0;	// number of macroscopic and microscopic bins, of overall particles and of the samples taken
	int interpolationref_init = interpolationref;		// gets potentially reset in between, but needs to be set to original setting, if several files are read one after another
	string suffix;										// end of name of read file

	int Nmic;											// total amount of microscopical sites
	int Nfull;											// number of sites in a certain state, that is currently read
	vector<double> xrange;								// discrete space
	vector<int> Eta_old;								// particle-distribution (based on rho_old)
	vector<double> Eta_old2;							// microscopical intitial distribution (as double)
	vector<double> eta_old;								// macroscopical intitial distribution
	vector<double> eta_old_int;							// interpolated eta_old
	vector<double> eta_new;								// macroscopical final distribution
	vector<double> eta_new_int;							// interpolated eta_new
	vector<double> rho_new;								// deterministic final distribution (only macroscopical resolution needed for the evaluation)
	vector<double> rho_new_int;							// interpolated rho_new
	vector<double> rho_old;								// deterministic initial condition after waiting time (i.e. in Y_old)
	vector<double> rho_old_int;							// interpolated rho_old
	vector<double> rho_old_soft;						// softened version of rho_old(_int) by local averaging over softener many positions
	vector<double> gamma_im;							// image of xrange over gamma (ordered accordingly)
	vector<double> allgamma_im;							// image of xrange over gamma for all x0 in x0range, one after another
	vector<double> allgammasq_im;						// derivative of allgamma_im
	vector<double>::iterator gamma_start;				// address of first element in allgamma_im for certain x0
	vector<double>::iterator gammasq_start;				// address of first element in allgammasq_im for certain x0
	vector<double> drhodT(lx0range);					// d rho/dt = (rho_new-eta_old)/dT measured by gamma centered at respective x0
	vector<double> sigmasample;							// storage for the sigmas of all the individual samples (store all sample-points of one x0 in a row, and then all sample-points of another x0 after it)
	vector<double> diametentsample;						// storage for the diagonal metric metries of all the individual entries
	vector<double> eta_a(lx0range);						// eta_old*gs.^2
	vector<double> rho_a(lx0range);						// rho_new*gs.^2
	vector<double> boundaries(2);						// ends of domain
	double Y_new = 0, Y_old = 0, sigma = -1, diametent = -1;	// initialize parameters for later
	int interpolationref_ref = interpolationref;		// interpolationrefinement reference for writing in results-file (needs to be memorized separately, to allow to reset interpolationref for const interpolationtype after having interpolated gamma, gammasq)

	int softener = 5, Nsoft;							// softener and amount of (effective) positions of softened state rho_old_soft

	string mode = "repeat";								// default-mode is "repeat"
	string repeattype = "keepinitial";					// default-mode is "keepinitial"

	string str;											// buffer for boring string-readings
	double value = -1;									// buffer for reading entries in state-vectors
	double x0 = -1, dx = 0, Dx = 0;						// x0-value, where filter-function gamma is concentrated
	int stateselector = 0;								// stateselector needed to remember which state in case of repeattype == "timesample"
	int J, j_stateno = 0, j_sigmasample, j_combinedsample = 0;	// initialize indices within statefile; initialize indices within sample-set, respectively; initialize indices of all combined readsamplings
	int statssize = 0, stateno;							// initialize length of resultstatistics (sigmasample, diametentsample); initialize the number of states in the file (of a respective filename)
	vector<int> samplesizesampling(readsampling);		// initializes the list of samplesizes (one for each of the files in namelist)

	string name;										// initialize name of file, that is currently read
	for( int j_rsampling=0; j_rsampling<readsampling; j_rsampling++ )	// go through all files in list
	{
		name = (*namelist).at(j_rsampling);				// get new filename, to be read

		J = 0;											// reset index within currently read state (just to be safe)
		j_stateno = 0;									// reset j_stateno, which counts how many states are already read from the current file
		j_sigmasample = 0;								// reset j_sigmasample, which counts how many measurements are arleady read from the current file
		if( samplingmode == "separate" )
		{
			j_combinedsample = 0;						// reset sample-amount, if samples are evaluated separately from the ones before
		}
		interpolationref = interpolationref_init;		// gets potentially reset in between if interpolationtype = const and interpolationref != 1

		// read file to find sigma, eta_old and rho_new:
		ifstream textfile2( name );						// open textfile for reading
		if( textfile2.is_open() )
		{
			cout << " Info - readagainC: Start reading \n " << name << "\n";
			// read headlines:
			textfile2 >> str >> dT >> str >> boundaries.at(0) >> boundaries.at(1) >> str >> Nbin >> str >> micres >> str >> Npart >> str >> samplesize >> str >> fsampling >> str >> sampling >> str >> wait >> str >> fwait >> str >> mode >> str >> repeattype >> str >> suffix;
			cout << "dT = " << dT << ", boundaries = " << boundaries.at(0) << ".." << boundaries.at(1) << ", Nbin = " << Nbin << ", micres = "<< micres << ", Npart = " << Npart << ", allsamples = " << sampling << "x" << fsampling << "x" << samplesize << ", wait = " << wait << ", mode = " << mode << ", repeattype = " << repeattype << ", suffix = " << suffix << "\n";
			cout << " refinement = " << refinement << "\n" << " diffusive scaling = " << diffscaling << ", \tepsilon = " << epsilon << "\n";
			cout << " interpolation = " << interpolationtype << " (" << interpolationref << ")\n";
			cout << " gamma = " << filtertype << " with ";
			for( int j_filterparameters=0; j_filterparameters<(*filterparameters_adr).size(); j_filterparameters++ )
			{
				cout << (*filterparameters_adr).at(j_filterparameters) << " ";
			}
			cout << "\n";

			// update statistic-lists dependent on samplingmode:
			if( samplingmode == "separate" )
			{
				statssize = 0;								// reset statistic/ delete memory
			}
			else if( samplingmode == "combined" )
			{
				statssize = sigmasample.size();				// keep current statistics
			}
			// redefine samplesize, such that it means the number of measurements in the current file:
			if( mode == "repeat" && repeattype == "statesample" )
			{
				samplesize = sampling*fsampling*samplesize;	// in case "statesample" is activated, the file contains also sampling and fsampling
			}
			sigmasample.resize(statssize + samplesize*lx0range);
			diametentsample.resize(statssize + samplesize*lx0range);
			samplesizesampling.at(j_rsampling) = samplesize;// current samplesize at this j_sampling (needed for final evaluation of average)

			// adjust stateno to repeattype: (from now on, "samplesize" stands for the amount of states to be read in the file)
			stateno = samplesize;
			if( mode == "repeat" )
			{
				if( repeattype == "timesample" )
				{
					stateno = 4*samplesize;				// for each sample, there are four states
				}
				else if( repeattype == "statesample" )
				{
					stateno = max(4,2*samplesize);
				}
			}

			// initialize state-variables:
			Nmic = Nbin*micres;								// total amount of microscopical sites
			Nint = Nbin*interpolationref;					// total amount of bins for internal interpolation, when calculating integrals etc
			Nsoft = Nmic/softener;
			Nfull = Nbin;
			Eta_old.resize(Nmic);
			Eta_old2.resize(Nmic);
			eta_old.resize(Nbin);
			rho_old.resize(Nbin);
			rho_old_int.resize(Nint);
			rho_old_soft.resize(Nsoft);
			eta_new.resize(Nbin);
			rho_new.resize(Nbin);
			eta_old_int.resize(Nint);
			eta_new_int.resize(Nint);
			rho_new_int.resize(Nint);
			if( diffscaling )
			{
				dt = dT/pow(Nmic,2.0);						// micro dT --> macro dT/sitenumber^2
			}
			else											// i.e. no diffusive scaling (macro- and micro-time coincide)
			{
				dt = dT;
			}

			// get space:
			xrange.resize(Nint);							// discrete space
			getxrange(&xrange,Nint,boundaries.at(0),boundaries.at(1));
			dx = xrange.at(1) - xrange.at(0);				// space-increment
			Dx = dx/refinement;								// microscopical increment for determining better resolution in derivatives (nabla gamma, dF^2/dx^2)
			gamma_im.resize(Nint);							// image of xrange over gamma (ordered accordingly)
			allgamma_im.resize(Nint*lx0range);
			allgammasq_im.resize(Nint*lx0range);

			// preset allgamma and allgammasq for all x0:
			for( int j_x0=0; j_x0<lx0range; j_x0++ )		// go through all x0 in range
			{
				x0 = x0range[j_x0];							// set current x0
				gamma(&gamma_im, &xrange, Nint, x0, filtertype, filterparameters_adr);	// get gamma_im for particular x0
				copy(gamma_im.begin(),gamma_im.end(),allgamma_im.begin()+j_x0*Nint);	// copy gamma_im into respective position of allgamma_im
			}

			vector<double> neighbours(2);					// initialize the two neighbours for the points in the domain
			vector<double> loc_gamma_im(2);					// initialize gamma(neighbours)
			for( int j_sq=0; j_sq<Nint*lx0range; j_sq++ )	// walk along space
			{
				if( refinement == 1 )
				{
					allgammasq_im.at(j_sq) = ( allgamma_im.at(j_sq-modulo(j_sq,Nint)+modulo(j_sq+1,Nint))-allgamma_im.at(j_sq-modulo(j_sq,Nint)+modulo(j_sq-1,Nint)) )/(2*dx);	// Euler-approx. for derivative (modulo for periodic boundary-conditions (otherwise not close to the boundary anyways))
				}
				else
				{
					neighbours.at(0) = xrange.at(j_sq)-Dx; neighbours.at(1) = xrange.at(j_sq)+Dx;	// update neighbours (this does not work, if gamma is not zero on the boundary and in a neighbourhood)
					gamma( &loc_gamma_im, &neighbours, 2, x0, filtertype, filterparameters_adr );	// update loc_gamma_im
					allgammasq_im.at(j_sq) = ( loc_gamma_im.at(1)-loc_gamma_im.at(0) )/(2.*Dx);		// Euler-approximation for derivative
				}
				allgammasq_im.at(j_sq) = pow( allgammasq_im.at(j_sq), 2.0 );						// square
			}

			if( interpolationtype=="const" && interpolationref!=1 )			// enough to interpolate allgamma, allgammasq, don't need to "refine" eta_old, eta_new 
			{
				for( int j_nonref=0; j_nonref<Nbin*lx0range; j_nonref++ )	// walk along contracted allgamma, allgammsq
				{
					allgamma_im.at(j_nonref) = accumulate(allgamma_im.begin()+j_nonref*interpolationref,allgamma_im.begin()+(j_nonref+1)*interpolationref,0.)/interpolationref;				// take average over interpolationref many entries and map to corresponding position in Nmic large allgamma_im
					allgammasq_im.at(j_nonref) = accumulate(allgammasq_im.begin()+j_nonref*interpolationref,allgammasq_im.begin()+(j_nonref+1)*interpolationref,0.)/interpolationref;
				}
				allgamma_im.resize(Nbin*lx0range);				// only keep the nonrefined length of allgamma_im (with the averaged values of the whole vector before) and discard the rest
				allgammasq_im.resize(Nbin*lx0range);			// note, that resize only shrinks, if not reallocated
				// also shrink xrange to "real" size:
				xrange.resize(Nbin);							// discrete space
				getxrange(&xrange,Nbin,boundaries.at(0),boundaries.at(1));
				dx = xrange.at(1) - xrange.at(0);				// space-increment
				Dx = dx/refinement;								// microscopical increment for determining better resolution in derivatives (nabla gamma, dF^2/dx^2)
				// also shrink "interpolated" states, as they don't need to be interpolated anymore:
				eta_old_int.resize(Nbin);
				rho_old_int.resize(Nbin);
				eta_new_int.resize(Nbin);
				rho_new_int.resize(Nbin);
				// reset sitenumber to Nbin rather than Nint to allow to use same formulas below:
				Nint = Nbin;
				interpolationref = 1;							// make sure the states don't get refined any further
			}

			// read Eta_old:
			J = 0;											// reset index within Eta_old2
			while( textfile2 >> value )						// always read until end of storagelength in units of Nmic (i.e. eta_old)
			{
				Eta_old.at(J) = value;
				J++;										// update index within Eta_old2
			}	// end of storagelength-while-loop
			// ignore x-delimiters:
			textfile2.clear();								// clear all flags of textfile, so no bitfail occurs
			textfile2.ignore();								// ignore 'x'
			// get eta_old:
			microtomacro(&Eta_old,&eta_old,Nbin,micres);	// get truely macroscopical eta_old for the final evaluation in getsigma
			interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_old to get eta_old_int
			if( mode == "repeat" && repeattype == "statesample" )
			{
				stateselector++;							// first state already read
				j_stateno++;
				// update rho_old:
				for( int j_rho=0; j_rho<Nbin; j_rho++ )
				{
					rho_old.at(j_rho) = ( rho_old.at(j_rho)*j_combinedsample + eta_old.at(j_rho) )/(j_combinedsample+1);
				}
			}
			else if( mode == "repeat" && repeattype == "localaverage" )
			{
				// get softened rho_old for evaluation of Y_old:
				microtomacro(&Eta_old,&rho_old_soft,Nsoft,micres*softener);
				interpolate(&rho_old_soft, &rho_old_int, Nsoft, interpolationref*softener,"const");
			}

			// for timer-output:
			clock_t t2 = clock();							// stop-time of timer

			// read states:
			while(!textfile2.eof())							// check if file is still not at end
			{
				J = 0;										// reset index within eta_new
				while( textfile2 >> value )					// always read until end of storagelength in units of Nmic (i.e. one eta_new)
				{
					if( mode == "continuous" || ( mode == "repeat" && (repeattype == "keepinitial") ) )
					{
						eta_new.at(J) = value;
					}
					else if( mode == "repeat" && repeattype == "timesample" )
					{
						if( stateselector==0 )
						{
							eta_old.at(J) = value;
						}
						else if( stateselector==1 )
						{
							rho_old.at(J) = value;
						}
						else if( stateselector==2 )
						{
							eta_new.at(J) = value;
						}
						else if( stateselector==3 )
						{
							rho_new.at(J) = value;
						}
					}
					else if( mode == "repeat" && repeattype == "statesample" )
					{
						if( stateselector==0 )
						{
							Eta_old.at(J) = value;
						}
						else if( stateselector==1 )
						{
							eta_new.at(J) = value;
						}
					}
					J++;									// update index within eta_new
					if( mode == "repeat" && "repeattype" == "statesample" && stateselector==0 )
					{
						Nfull = Nmic;
					}
					else
					{
						Nfull = Nbin;
					}
					if( J==Nfull )							// i.e. have read one complete eta_new (note, that J is already updated for next measurement)
					{
						// update stateselector:
						if( mode == "repeat" && repeattype == "timesample" )
						{
							stateselector = modulo(stateselector+1,4);
						}
						else if( mode == "repeat" && repeattype == "statesample" )
						{
							stateselector = modulo(stateselector+1,2);
						}

						// evaluate states:
						if( mode == "repeat" && repeattype == "keepinitial" )	// need second evaluation
						{
							// update rho_new:
							for( int j_rho=0; j_rho<Nbin; j_rho++ )
							{
								rho_new.at(j_rho) = ( rho_new.at(j_rho)*j_combinedsample + eta_new.at(j_rho) )/(j_combinedsample+1);
							}
						}
						else if ( mode == "repeat" && repeattype == "statesample" )
						{
							if( stateselector==1 )	// note: stateselector already updated for next reading
							{
								microtomacro(&Eta_old,&eta_old,Nbin,micres);	// get truely macroscopical eta_old for the final evaluation in getsigma
								// update rho_old:
								for( int j_rho=0; j_rho<Nbin; j_rho++ )
								{
									rho_old.at(j_rho) = ( rho_old.at(j_rho)*j_combinedsample + eta_old.at(j_rho) )/(j_combinedsample+1);
								}
							}
							else if( stateselector==0 )	// note: stateselector already updated for next reading
							{
								// update rho_new:
								for( int j_rho=0; j_rho<Nbin; j_rho++ )
								{
									rho_new.at(j_rho) = ( rho_new.at(j_rho)*j_combinedsample + eta_new.at(j_rho) )/(j_combinedsample+1);
								}
							}
						}
						else if( mode == "continuous" || (mode == "repeat" && (repeattype == "purelystochastic" || ( repeattype == "timesample" && stateselector==0 ))) )	// can evaluate right away; note: stateselector already updated
						{
							// evaluate states to get sigma:
							interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);	// interpolate eta_new for evaluation
							if( mode == "repeat" && repeattype == "timesample" )	// mode dominates repeattype; already checked that we are at end of state-cycle (i.e. sateselector==0)
							{
								interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);
								interpolate(&rho_old, &rho_old_int, Nbin, interpolationref, interpolationtype);
								interpolate(&rho_new, &rho_new_int, Nbin, interpolationref, interpolationtype);
							}
							for( int j_x0=0; j_x0<lx0range; j_x0++ )	// go through all x0 in range
							{
								x0 = x0range[j_x0];						// set current x0
								gamma_start = allgamma_im.begin()+Nint*j_x0;
								gammasq_start = allgammasq_im.begin()+Nint*j_x0;
								if( mode == "continuous" || (mode == "repeat" && repeattype == "purelystochastic") )
								{
									Y_old = 0;
									Y_new = getY(&eta_old_int, &eta_new_int, gamma_start, Nint, epsilon);
								}
								else if( mode == "repeat" && repeattype == "timesample" )
								{
									Y_old = getY(&eta_old_int, &rho_old_int, gamma_start, Nint, epsilon);
									Y_new = getY(&eta_new_int, &rho_new_int, gamma_start, Nint, epsilon);
								}
								sigma = getsigma( 0, Y_new-Y_old, &eta_old_int, dt, gammasq_start, Dx );
								diametent = getF((Y_new-Y_old)/sqrt(double(Nmic))) / dt;	// "/sqrt(double(Nmic))" as states in the code are non-normalized (and epsilon = 1 to compensate), but in theory they are (and epsilon = 1/sqrt(Nmic))
								sigmasample.at(statssize + j_x0*samplesize+j_sigmasample) = sigma;
								diametentsample.at(statssize + j_x0*samplesize+j_sigmasample) = diametent;
								if( j_sigmasample==0 || ( mode == "repeat" && repeattype == "timesample" && j_sigmasample == 3 ) )	// i.e. only the first time (j_sigmasample not updated yet, so "3" refers to fourth read state)
								{
									eta_a.at(j_x0) = getrho_a(&eta_old_int, Nint, gammasq_start, interpolationref);
								}
								else if( j_sigmasample==samplesize-1  )	// write very final state as rho_a
								{
									rho_a.at(j_x0) = getrho_a(&eta_new_int, Nint, gammasq_start, interpolationref);
									// calculate drhodT from very final state:
									if( mode == "continuous" || repeattype == "purelystochastic" )
									{
										drhodT.at(j_x0) = getY( &rho_new_int, &eta_old_int, gamma_start, Nint, pow(Nmic*samplesize*dt,2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are; should actually happen for every j_sigmasample
									}
									else if( repeattype == "timesample" )
									{
										drhodT.at(j_x0) = getY( &rho_new_int, &rho_old_int, gamma_start, Nint, pow(Nmic*samplesize*dt,2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are; should actually happen for every j_sigmasample
									}
								}
							}	// end x0-range
							if( mode == "continuous" )												// mode is dominant over repeattype
							{
								copy(eta_new_int.begin(), eta_new_int.end(), eta_old_int.begin());	// update: future old state is current new state
							}
							else if( mode == "repeat" && repeattype == "purelystochastic" )
							{
								// don't change eta_old_int
							}
						}	// end mode-specific evaluations
						j_stateno++;						// just read new state
						
						// update amount of measurements currently contributing to result:
						if( stateselector==0 )				// note: stateselector was already updated - so this means, that a new measurement will be started to be read in the next cycle
						{
							j_combinedsample++;
							j_sigmasample++;				// update sigma-sample-index for reading next eta_new
						}

						// reset J to refill eta_new with new one:
						J = 0;
						
						// for control-window-progress-feedback:
						if( stateno>=10 )
						{
							if( modulo(j_stateno,int(0.1*stateno)) < modulo(j_stateno-1,int(0.1*stateno)) )
							{
								cout << ".." << (100.*j_stateno)/stateno<< "%.";
								if(j_stateno+1 < 0.15*stateno)
								{
									clock_t t3 = clock();	// stop-time of timer
									cout << "(" << double(t3-t2)/ CLOCKS_PER_SEC << ")";
								}
							}
							if( j_stateno==stateno )
							{
								cout << "\n";
							}
						}
					}	// end of eta_new-evaluation
				}	// end of storagelength-while-loop
				// ignore x-delimiters:
				textfile2.clear();							// clear all flags of textfile, so no bitfail occurs
				textfile2.ignore();							// ignore 'x'
			}	// end of "end of file"-while-loop
			textfile2.close();								// close textfile again
		}	// end of file-open-if
		else												// sth went wrong
		{
			cout << "  Warning - readagainC: Could not open a textfile where states are safed!\n(" << name << ")\n";
			return;
		}

		// read file again, to find sigma:
		if( mode == "repeat" && (repeattype == "keepinitial" || repeattype == "statesample") )
		{
			// for timer-output:
			clock_t t2 = clock();								// stop-time of timer
			j_stateno = 0;
			j_sigmasample = 0;									// reset index within file
			j_combinedsample = 0;								// reset index within sigmasample
			stateselector = 0;									// reset stateselector
			// interpolate rho_new to get rho_new_int:
			interpolate(&rho_new, &rho_new_int, Nbin, interpolationref, interpolationtype);
			if( repeattype == "statesample" )					// also care about rho_old
			{
				interpolate(&rho_old, &rho_old_int, Nbin, interpolationref, interpolationtype);
			}
			
			bool foundstoragelength = 0;						// set to "1", as soon as first "x" is hit
			int samplepos = -1;
			ifstream textfile( name );							// open textfile for reading
			if( textfile.is_open() )
			{
				// read headlines:
				textfile >> str >> dT >> str >> boundaries.at(0) >> boundaries.at(1) >> str >> Nbin >> str >> micres >> str >> Npart >> str >> samplesize >> str >> fsampling >> str >> sampling >> str >> wait >> str >> fwait >> str >> mode >> str >> repeattype >> str >> suffix;
				// redefine samplesize, such that it means the number of measurements in the current file:
				if( mode == "repeat" && repeattype == "statesample" )
				{
					samplesize = sampling*fsampling*samplesize;	// in case "statesample" is activated, the file contains also sampling and fsampling
				}

				// read Eta_old:
				J = 0;											// reset index within Eta_old2
				while( textfile >> value )						// always read until end of storagelength in units of Nmic (i.e. eta_old)
				{
					if( repeattype == "keepinitial" )
					{
						// don't do anything, as we don't care about the old state (it's the same as rho_old anyways)
					}
					else	// i.e. repeattype == "statesample"
					{
						Eta_old.at(J) = value;
						J++;	
					}
				}	// end of storagelength-while-loop
				// ignore x-delimiters:
				textfile.clear();								// clear all flags of textfile, so no bitfail occurs
				textfile.ignore();								// ignore 'x'
				// transform Eta_old2 into eta_old_int as evaluated by getY, getsigma etc.:
				if( repeattype == "statesample" )				// also care about rho_old
				{
					microtomacro(&Eta_old,&eta_old,Nbin,micres);// get truely macroscopical eta_old for the final evaluation in getsigma
					interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);
					stateselector = 1;							// now ready to read eta_new
					j_stateno++;
				}

				// read states:
				while(!textfile.eof())							// check if file is still not at end
				{
					J = 0;										// reset index within eta_new
					while( textfile >> value )					// always read until end of storagelength in units of Nmic (i.e. one eta_new)
					{
						if( repeattype == "statesample" && stateselector==0 ) 
						{
							Eta_old.at(J) = value;
							Nfull = Nmic;
						}
						else
						{
							eta_new.at(J) = value;
							Nfull = Nbin;
						}
						J++;									// update index within eta_new
						if( J==Nfull )							// i.e. have read one complete eta_new (note, that J is already updated for next measurement)
						{
							// update stateselector:
							if( repeattype == "statesample" )
							{
								stateselector = modulo(stateselector+1,2);	// ...for next reading
							}
							// get sigma:
							if( repeattype == "keepinitial" || (repeattype == "statesample" && stateselector==0) )
							{
								// interpolate read state:
								interpolate(&eta_new, &eta_new_int, Nbin, interpolationref, interpolationtype);
								if( repeattype == "statesample" )
								{
									microtomacro(&Eta_old,&eta_old,Nbin,micres);	// get truely macroscopical eta_old for the final evaluation in getsigma
									interpolate(&eta_old, &eta_old_int, Nbin, interpolationref, interpolationtype);
								}
								// go through space and determine diffusivities:
								for( int j_x0=0; j_x0<lx0range; j_x0++ )	// go through all x0 in range
								{
									x0 = x0range[j_x0];						// set current x0
									samplepos = statssize + j_x0*samplesize+j_sigmasample;
									gamma_start = allgamma_im.begin()+Nint*j_x0;
									gammasq_start = allgammasq_im.begin()+Nint*j_x0;
									// evaluate sigma:
									if( repeattype == "keepinitial" )
									{
										Y_old = 0;
										Y_new = getY(&rho_new_int, &eta_new_int, gamma_start, Nint, epsilon);
										sigma = getsigma( 0, Y_new-Y_old, &eta_old_int, dt, gammasq_start, Dx );
									}
									else	// i.e. statesample is active
									{
										Y_old = getY(&rho_old_int, &eta_old_int, gamma_start, Nint, epsilon);
										Y_new = getY(&rho_new_int, &eta_new_int, gamma_start, Nint, epsilon);
										sigma = getsigma( 0, Y_new-Y_old, &rho_old_int, dt, gammasq_start, Dx );
									}
									diametent = getF((Y_new-Y_old)/sqrt(double(Nmic))) / dt;	// "/sqrt(double(Nmic))" as states in the code are non-normalized (and epsilon = 1 to compensate), but in theory they are (and epsilon = 1/sqrt(Nmic))
									sigmasample.at(samplepos) = sigma;
									diametentsample.at(samplepos) = diametent;
									if( j_sigmasample==0 )
									{
										// get average densities:
										if( repeattype == "keepinitial" )
										{
											eta_a.at(j_x0) = getrho_a(&eta_old_int, Nint, gammasq_start, interpolationref);
											// get del_t rho (i.e. lhs of gradient-flow):
											drhodT.at(j_x0) = getY( &rho_new_int, &eta_old_int, gamma_start, Nint, pow(Nmic*dt,2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are
										}
										else	// i.e. repeattype == "statesample"
										{
											eta_a.at(j_x0) = getrho_a(&rho_old_int, Nint, gammasq_start, interpolationref);
											// get del_t rho (i.e. lhs of gradient-flow):
											drhodT.at(j_x0) = getY( &rho_new_int, &rho_old_int, gamma_start, Nint, pow(Nmic*dt,2.0) );	// "Nmic*" as states for the code are non-normalized, but for theory they are
										}
										rho_a.at(j_x0) = getrho_a(&rho_new_int, Nint, gammasq_start, interpolationref);
									}
								}
							}
							j_stateno++;

							// update amount of measurements currently contributing to result:
							if( stateselector==0 )				// note: stateselector was already updated - so this means, that a new measurement will be started to be read in the next cycle
							{
								j_combinedsample++;
								j_sigmasample++;				// update sigma-sample-index for reading next eta_new
							}

							// reset J to refill eta_new with new one:
							J = 0;

							// for control-window-progress-feedback:
							if( stateno>=10 )
							{
								if( modulo(j_stateno,int(0.1*stateno)) < modulo(j_stateno-1,int(0.1*stateno)) )
								{
									cout << ".." << (100.*j_stateno)/stateno << "%.";
									if(j_stateno+1 < 0.15*stateno)
									{
										clock_t t3 = clock();	// stop-time of timer
										cout << "(" << double(t3-t2)/ CLOCKS_PER_SEC << ")";
									}
								}
								if( j_stateno==stateno )
								{
									cout << "\n";
								}
							}
						}
					}	// end of storagelength-while-loop
					// ignore x-delimiters:
					textfile.clear();							// clear all flags of textfile, so no bitfail occurs
					textfile.ignore();							// ignore 'x'
				}	// end of "end of file"-while-loop
				textfile.close();								// close textfile again
			}	// end of file-open-if
			else												// sth went wrong
			{
				cout << "  Warning - readagainC: Could not open a textfile to save state!\n(" << name << ")\n";
			}
		}	// end if repeat

		if( samplingmode == "separate" )
		{
			// for output in controlwindow:
			for( int j_x0=0; j_x0<lx0range; j_x0++ )
			{
				average.at(j_x0) = accumulate(j_x0*samplesize+sigmasample.begin(), (j_x0+1)*samplesize+sigmasample.begin(), 0.)/samplesize;
				stdderr.at(j_x0) = sqrt( ( inner_product(j_x0*samplesize+sigmasample.begin(), (j_x0+1)*samplesize+sigmasample.begin(), j_x0*samplesize+sigmasample.begin(), 0.)/samplesize - pow(average.at(j_x0),2.) )/(samplesize-1) );
				cout << " sigma(rho("<< x0range[j_x0] <<")) = " << "sigma("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << average.at(j_x0) << "  +-  " << stdderr.at(j_x0) << "\n";
	
				cout << " <gamma,drho/dT> = " << drhodT[j_x0] << "\n";
				avdiametent.at(j_x0) = accumulate(j_x0*samplesize+diametentsample.begin(), (j_x0+1)*samplesize+diametentsample.begin(), 0.)/samplesize;
				stderr_diametent.at(j_x0) = sqrt( ( inner_product(j_x0*samplesize+diametentsample.begin(), (j_x0+1)*samplesize+diametentsample.begin(), j_x0*samplesize+diametentsample.begin(), 0.)/samplesize - pow(avdiametent.at(j_x0),2.) )/(samplesize-1) );
				cout << " diametent(rho("<< x0range[j_x0] <<")) = " << "diametent("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << avdiametent.at(j_x0) << "  +-  " << stderr_diametent.at(j_x0) << "\n";
			}	// end x0-loop

			// to save the calculated data in "C++ read results":
			saveresultstotext( x0range, lx0range, refinement, diffscaling, epsilon, &eta_a, &rho_a, &average, &stdderr, &drhodT, &avdiametent, &stderr_diametent, name, t1, storagelength, filtertype, filterparameters_adr, interpolationtype, interpolationref_ref, mode );
		}
		cout << "\n\n";
	}	// end name readsampling-forloop

	if( samplingmode == "combined" )
	{
		int totalsamples;
		// for output in controlwindow:
		for( int j_x0=0; j_x0<lx0range; j_x0++ )
		{
			statssize = 0;
			for( int j_rsampling=0; j_rsampling<readsampling; j_rsampling++ )
			{
				average.at(j_x0) += accumulate(j_x0*samplesizesampling.at(j_rsampling)+statssize+sigmasample.begin(), (j_x0+1)*samplesizesampling.at(j_rsampling)+statssize+sigmasample.begin(), 0.);
				statssize += samplesizesampling.at(j_rsampling)*lx0range;
			}
			if( j_x0==0 )	// totalsamples is supposed to be the same for all x0 as always the same x0range
			{
				totalsamples = accumulate(samplesizesampling.begin(), samplesizesampling.end(), 0);
			}
			average.at(j_x0) = average.at(j_x0) / double(totalsamples);
			statssize = 0;
			for( int j_rsampling=0; j_rsampling<readsampling; j_rsampling++ )
			{
				stdderr.at(j_x0) += inner_product(j_x0*samplesizesampling.at(j_rsampling)+statssize+sigmasample.begin(), (j_x0+1)*samplesizesampling.at(j_rsampling)+statssize+sigmasample.begin(), j_x0*samplesizesampling.at(j_rsampling)+statssize+sigmasample.begin(), 0.);
				statssize += samplesizesampling.at(j_rsampling)*lx0range;
			}
			stdderr.at(j_x0) = sqrt( stdderr.at(j_x0)/double(totalsamples) - pow(average.at(j_x0),2.) )/double(totalsamples-1);
			cout << " sigma(rho("<< x0range[j_x0] <<")) = " << "sigma("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << average.at(j_x0) << "  +-  " << stdderr.at(j_x0) << "\n";
	
			cout << " <gamma,drho/dT> = " << drhodT[j_x0] << "\n";
			statssize = 0;
			for( int j_rsampling=0; j_rsampling<readsampling; j_rsampling++ )
			{
				avdiametent.at(j_x0) += accumulate(j_x0*samplesizesampling.at(j_rsampling)+statssize+diametentsample.begin(), (j_x0+1)*samplesizesampling.at(j_rsampling)+statssize+diametentsample.begin(), 0.);
				statssize += samplesizesampling.at(j_rsampling)*lx0range;
			}
			avdiametent.at(j_x0) = avdiametent.at(j_x0) / double(totalsamples);
			
			statssize = 0;
			for( int j_rsampling=0; j_rsampling<readsampling; j_rsampling++ )
			{
				stderr_diametent.at(j_x0) += inner_product(j_x0*samplesizesampling.at(j_rsampling)+statssize+diametentsample.begin(), (j_x0+1)*samplesizesampling.at(j_rsampling)+statssize+diametentsample.begin(), j_x0*samplesizesampling.at(j_rsampling)+statssize+diametentsample.begin(), 0.);
				statssize += samplesizesampling.at(j_rsampling)*lx0range;
			}
			stderr_diametent.at(j_x0) = sqrt( stderr_diametent.at(j_x0)/double(totalsamples) - pow(avdiametent.at(j_x0),2.) ) / double(totalsamples-1);
			cout << " diametent(rho("<< x0range[j_x0] <<")) = " << "diametent("<< eta_a[j_x0] << "=" << rho_a[j_x0] <<") = \t" << avdiametent.at(j_x0) << "  +-  " << stderr_diametent.at(j_x0) << "\n";
		}	// end x0-loop

		// to save the calculated data in "C++ read results":
		saveresultstotext( x0range, lx0range, refinement, diffscaling, epsilon, &eta_a, &rho_a, &average, &stdderr, &drhodT, &avdiametent, &stderr_diametent, name, t1, storagelength, filtertype, filterparameters_adr, interpolationtype, interpolationref_ref, mode );
	}	// end samplingmode==combined

	return;
}