import sys
import os
from pathlib import Path
os.environ['TF_CPP_MIN_LOG_LEVEL'] = '3'

import tensorflow as tf
import numpy as np

NUM_PHASES = 4
NUM_STYLES = 3
X_SIZE = 246
X_STYLE_START = 48

ENC_SIZE = 2
ENC_LAYERS = [100, 50, 10]
REGULARIZATION_L1L2 = (0., 0.) # (0.01, 0.01)
DROPOUT = 0. # 0.3
BATCH_SIZE = 32
NUM_EPOCHS = 100
SHUFFLE_BUFFER_SIZE = 10_000

def main(tfrecord_path, log_path):
    log_path = Path(log_path)
    log_path.mkdir(parents=True, exist_ok=True)
    with tf.Graph().as_default(), tf.Session() as sess:
        tf.set_random_seed(0)
        np.random.seed(0)
        model = make_model()
        with Path(log_path, 'summary.txt').open('w') as f:
            model.summary(print_fn=lambda s: print(s, file=f))
        with tf.Graph().as_default():
            train_ds = get_data(make_dataset(Path(tfrecord_path, 'train')))
            test_ds = get_data(make_dataset(Path(tfrecord_path, 'test')))
        model.fit(*train_ds, epochs=NUM_EPOCHS, batch_size=BATCH_SIZE, shuffle=True,
                  validation_data=test_ds,
                  callbacks=[tf.keras.callbacks.TensorBoard(Path(log_path, 'log'), batch_size=BATCH_SIZE)])
        gd = freeze_session(sess, output_names=['Encoded'])
        tf.train.write_graph(gd, str(log_path), 'encoder.tf', as_text=False)
        gd = freeze_session(sess, output_names=['Encoded', 'Decoded'])
        tf.train.write_graph(gd, str(log_path), 'autoencoder.tf', as_text=False)
        print('Done')

def make_model():
    reg = tf.keras.regularizers.L1L2(*REGULARIZATION_L1L2)
    input_size = NUM_PHASES + NUM_STYLES
    x = tf.keras.Input(shape=(input_size,), name='Input')
    y = x
    y = tf.keras.layers.Dropout(DROPOUT)(y)
    for i, h in enumerate(ENC_LAYERS, 1):
        y = tf.keras.layers.Dense(h, activation=tf.nn.relu, kernel_regularizer=reg, name=f'Encoder_{i}')(y)
        y = tf.keras.layers.Dropout(DROPOUT)(y)
    y = tf.keras.layers.Dense(ENC_SIZE, activation=tf.nn.sigmoid, kernel_regularizer=reg, name='Code')(y)
    tf.identity(y, name='Encoded')
    for i, h in enumerate(reversed(ENC_LAYERS), 1):
        y = tf.keras.layers.Dense(h, activation=tf.nn.relu, kernel_regularizer=reg, name=f'Decoder_{i}')(y)
        y = tf.keras.layers.Dropout(DROPOUT)(y)
    y = tf.keras.layers.Dense(input_size, activation=tf.nn.sigmoid, kernel_regularizer=reg, name='Decode')(y)
    tf.identity(y, name='Decoded')
    model = tf.keras.Model(inputs=x, outputs=y)
    model.compile(optimizer='Adam', loss='MSE')
    return model

def make_dataset(tfrecord_path, shuffle=False, repeat=False):
    dataset = (tf.data.Dataset
        .list_files(str(Path(tfrecord_path, '*.tfrecord')))
        .interleave(tf.data.TFRecordDataset, cycle_length=4, block_length=16, num_parallel_calls=4)
        .map(parse_style_phase)
        .cache()
        )
    if repeat:
        dataset = dataset.repeat()
    if shuffle:
        dataset = dataset.shuffle(SHUFFLE_BUFFER_SIZE, seed=0, reshuffle_each_iteration=True)
    dataset = (dataset
        .batch(BATCH_SIZE)
        .map(lambda s, p: 2 * (tf.concat([p, s], axis=1),))
        .prefetch(1)
        )
    return dataset

def get_data(dataset):
    with tf.Session() as sess:
        x, y = dataset.make_one_shot_iterator().get_next()
        xs, ys = [], []
        try:
            while True:
                xv, yv = sess.run((x, y))
                xs.append(xv)
                ys.append(yv)
        except tf.errors.OutOfRangeError: pass
        xs = np.concatenate(xs, axis=0)
        ys = np.concatenate(ys, axis=0)
        return xs, ys

def parse_style_phase(example_proto):
    features = {'x': tf.FixedLenFeature((X_SIZE,), tf.float32),
                'phase': tf.FixedLenFeature((NUM_PHASES,), tf.float32)}
    parsed_features = tf.parse_single_example(example_proto, features)
    return parsed_features['x'][X_STYLE_START:X_STYLE_START + NUM_STYLES], parsed_features['phase']

def freeze_session(session, keep_var_names=None, output_names=None, clear_devices=True):
    """
    Freezes the state of a session into a pruned computation graph.

    Creates a new computation graph where variable nodes are replaced by
    constants taking their current value in the session. The new graph will be
    pruned so subgraphs that are not necessary to compute the requested
    outputs are removed.
    @param session The TensorFlow session to be frozen.
    @param keep_var_names A list of variable names that should not be frozen,
                          or None to freeze all the variables in the graph.
    @param output_names Names of the relevant graph outputs.
    @param clear_devices Remove the device directives from the graph for better portability.
    @return The frozen graph definition.
    """
    graph = session.graph
    with graph.as_default():
        freeze_var_names = list(set(v.op.name for v in tf.global_variables()).difference(keep_var_names or []))
        output_names = output_names or []
        output_names += [v.op.name for v in tf.global_variables()]
        input_graph_def = graph.as_graph_def()
        if clear_devices:
            for node in input_graph_def.node:
                node.device = ""
        frozen_graph = tf.graph_util.convert_variables_to_constants(
            session, input_graph_def, output_names, freeze_var_names)
        return frozen_graph

if __name__ == '__main__':
    main(*sys.argv[1:])
