# Analysis for "Affect Recognition using Psychophysiological Correlates in High Intensity VR Exergaming"
# CHI 2020 paper by Soumya C. Barathi, Michael Proulx, Eamonn O'Neill, Christof Lutteroth
# Please cite the paper if you find the analysis below useful. Thanks!

# install the packages below in RStudio using Tools -> Install
library(reshape2)
library(dplyr)
library(nlme)
library(tidyverse)
library(rmcorr)
library(psych)
library(plyr)
library(lme4)

dw = read.csv("Experiment_2.csv")

# wide to long conversion for IMI Interest/Enjoyment columns
d1 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
            "Ow.IMI.Enj", "Uw.IMI.Enj", "Opt.IMI.Enj"), 
          variable.name="Condition",
          value.name="Interest")
d1$Condition <- as.character(d1$Condition)
d1$Condition[d1$Condition=="Ow.IMI.Enj"] <- "Ow"
d1$Condition[d1$Condition=="Uw.IMI.Enj"] <- "Uw"
d1$Condition[d1$Condition=="Opt.IMI.Enj"] <- "Opt"

# wide to long conversion for Power columns
d2 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
            "Performance.U", "Performance.Ow", "Performance.Opt"), 
          variable.name="Condition",
          value.name="Power")
d2$Condition <- as.character(d2$Condition)
d2$Condition[d2$Condition=="Performance.Ow"] <- "Ow"
d2$Condition[d2$Condition=="Performance.U"] <- "Uw"
d2$Condition[d2$Condition=="Performance.Opt"] <- "Opt"

# wide to long conversion for Conductivity columns
d3 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
            "Ow.SkCond", "Uw.SkCond", "Opt.SkCond"), 
          variable.name="Condition",
          value.name="SkCond")
d3$Condition <- as.character(d3$Condition)
d3$Condition[d3$Condition=="Ow.SkCond"] <- "Ow"
d3$Condition[d3$Condition=="Uw.SkCond"] <- "Uw"
d3$Condition[d3$Condition=="Opt.SkCond"] <- "Opt"

# wide to long conversion for Blinks columns
d4 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
            "Ow.Bpm", "Uw.Bpm", "Opt.Bpm"), 
          variable.name="Condition",
          value.name="Bpm")
d4$Condition <- as.character(d4$Condition)
d4$Condition[d4$Condition=="Ow.Bpm"] <- "Ow"
d4$Condition[d4$Condition=="Uw.Bpm"] <- "Uw"
d4$Condition[d4$Condition=="Opt.Bpm"] <- "Opt"

# wide to long conversion for Affect columns
d5 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
            "Ow.Av.ES", "Uw.Avg.ES", "Opt.Av.ES"), 
          variable.name="Condition",
          value.name="ES")
d5$Condition <- as.character(d5$Condition)
d5$Condition[d5$Condition=="Ow.Av.ES"] <- "Ow"
d5$Condition[d5$Condition=="Uw.Avg.ES"] <- "Uw"
d5$Condition[d5$Condition=="Opt.Av.ES"] <- "Opt"

# wide to long conversion for Pupil columns
d6 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
                  "Dil.Ow.Av", "Dil.Uw.Avg", "Dil.Opt.Av"), 
          variable.name="Condition",
          value.name="Dil")
d6$Condition <- as.character(d6$Condition)
d6$Condition[d6$Condition=="Dil.Ow.Av"] <- "Ow"
d6$Condition[d6$Condition=="Dil.Uw.Avg"] <- "Uw"
d6$Condition[d6$Condition=="Dil.Opt.Av"] <- "Opt"

# wide to long conversion for Fixations columns
d7 = melt(dw, 
          id.vars=c("P.No"), 
          measure.vars=c(
                  "Ow.Gaze.Hits", "Uw.Gaze.Hits", "Opt.Gaze.Hits"), 
          variable.name="Condition",
          value.name="Gaze")
d7$Condition <- as.character(d7$Condition)
d7$Condition[d7$Condition=="Ow.Gaze.Hits"] <- "Ow"
d7$Condition[d7$Condition=="Uw.Gaze.Hits"] <- "Uw"
d7$Condition[d7$Condition=="Opt.Gaze.Hits"] <- "Opt"

# merge all the newly created long tables together into a single one called d
d = merge(d1, d2)
d = merge(d, d3)
d = merge(d, d4)
d = merge(d, d5)
d = merge(d, d6)
d = merge(d, d7)
d$Condition <- as.factor(d$Condition)

# Normalise variables using standard z-score transforms,
# i.e. dividing by a participant's standard deviation
# and centering at the participant's mean (except for Conductivity)
d = ddply(d, c("P.No"), transform, GazeN = scale(Gaze))
d = ddply(d, c("P.No"), transform, DilN = scale(Dil))
d = ddply(d, c("P.No"), transform, SkCondN = scale(SkCond, center=FALSE))


# Correlation Analysis

# Note: rmcorr calculates two-tailed p-values, but the hypotheses below
# are directed so require one-tailed p-values. The one-tailed p-values in
# the paper are calculated by halving the corresponding two-tailed p-values.

# Affect is a valid measure, highly correlated with IMI Interest/Enjoyment
# Testing hypothesis H4, RmCorr for Affect and IMI Enjoy
rmcorr(P.No, d$Interest, d$ES, d)  # r=0.579

# Correlates of Affect
# RmCorr for Conductivity and Affect
rmcorr(P.No, SkCondN, ES, d) # r=-0.059
# Testing hypothesis H5, RmCorr for Blinks and Affect
rmcorr(P.No, Bpm, ES, d) # r=-0.374
# Testing hypothesis H6, RmCorr for Pupil and Affect
rmcorr(P.No, DilN, ES, d) # r=0.346
# Testing hypothesis H7, RmCorr for Fixations and Affect
rmcorr(P.No, GazeN, ES, d) # r=0.409
# Testing hypothesis H8, RmCorr for Power and Affect
rmcorr(P.No, Power, ES, d) # r=0.382

# Correlates of Valence
# RmCorr for Conductivity and sign(Affect)
rmcorr(P.No, SkCondN, sign(ES), d) # r=-0.092
# RmCorr for Blinks and sign(Affect)
rmcorr(P.No, Bpm, sign(ES), d) # r=-0.460 
# RmCorr of Pupil and sign(Affect)
rmcorr(P.No, DilN, sign(ES), d) # r=0.270
# RmCorr of Fixations and sign(Affect)
rmcorr(P.No, GazeN, sign(ES), d) # r=0.512
# RmCorr of Power and sign(Affect)
rmcorr(P.No, Power, sign(ES), d) # r=0.296

# Correlates of Arousal
# Testing hypothesis H9, RmCorr for Conductivity and abs(Affect)
rmcorr(P.No, SkCondN, abs(ES), d) # r=.335
# RmCorr for Blinks and abs(Affect)
rmcorr(P.No, Bpm, abs(ES), d) # r=0.121
# RmCorr of Pupil and abs(Affect)
rmcorr(P.No, DilN, abs(ES), d) # r=-0.211
# RmCorr of Fixations and abs(Affect)
rmcorr(P.No, GazeN, abs(ES), d) # r=-0.115
# RmCorr of Power and abs(Affect)
rmcorr(P.No, Power, abs(ES), d) # r=0.116


# Regression Analysis

# Note: Similar to rmcorr, gls calculates two-tailed p-values. Since 
# the tests below are based on our directed hypotheses, one-tailed 
# p-values are required. The one-tailed p-values for the coefficients 
# in the paper are calculated by halving the corresponding two-tailed 
# p-values provided by gls below.

# Effects of Fixations, Pupil, Blinks and Power on Affect
d1 = d
d1 <- groupedData(ES ~ DilN + Bpm + Power + GazeN | P.No, data = d1)
fit <- gls(ES ~ DilN + Bpm + Power + GazeN, data = d1,
           corr = corCompSymm(form= ~ 1 | P.No) )
summary(fit)
intervals(fit)
# Coefficients:
#              Value     Std.Error  t-value    p-value
# (Intercept)  0.5344096 0.5776291  0.9251778  0.3594
# DilN         0.3222614 0.1678733  1.9196701  0.0607
# Bpm          0.0015887 0.0019359  0.8206486  0.4158
# Power       -0.0009866 0.0014874 -0.6633460  0.5102
# GazeN        0.4144177 0.1682315  2.4633775  0.0173
# Degrees of freedom: 54 total; 49 residual
# Coefficients:
#             lower         est.         upper
# (Intercept) -0.626379543  0.5344096001 1.695198744
# DilN        -0.015092669  0.3222613730 0.659615415
# Bpm         -0.002301584  0.0015886527 0.005478889
# Power       -0.003975638 -0.0009866448 0.002002349
# GazeN        0.076343837  0.4144176939 0.752491551

# calculate effect size R^2 of linear model based on rmcorr
rmcorr(P.No, 0.3222614*DilN + 0.4144177*GazeN, ES, d)  # r=0.496, R^2=0.246


# Effects of Fixations, Blinks and Power on Valence
d1 = d
d1 <- groupedData(sign(ES) ~ Power + Bpm + GazeN 
                  | P.No, data = d1)
fit <- gls(sign(ES) ~ Power + Bpm + GazeN, data = d1,
           corr = corCompSymm(form= ~ 1 | P.No) )
summary(fit)
intervals(fit)
# Coefficients:
#              Value     Std.Error  t-value   p-value
# (Intercept)  0.7153859 0.4437519  1.612130  0.1132
# Power       -0.0011587 0.0011441 -1.012765  0.3160
# Bpm          0.0000069 0.0014828  0.004627  0.9963
# GazeN        0.4558948 0.1294688  3.521272  0.0009
# Degrees of freedom: 54 total; 50 residual
# Coefficients:
#             lower         est.         upper
# (Intercept) -0.175916015  7.153859e-01 1.606687912
# Power       -0.003456587 -1.158667e-03 0.001139252
# Bpm         -0.002971417  6.861229e-06 0.002985139
# GazeN        0.195849102  4.558948e-01 0.715940412

# calculate effect size R^2 of linear model based on rmcorr
rmcorr(P.No, GazeN, sign(ES), d) # r=0.512, R^2=0.262


# Effects of sign(Fixations)Conductivity, Pupil and Fixations on Affect
d$GazeNXSkCondN = sign(d$GazeN) * d$SkCondN
d1 = d
d1 <- groupedData(ES ~ GazeNXSkCondN + DilN + GazeN
                  | P.No, data = d1)
fit <- gls(ES ~ GazeNXSkCondN + DilN + GazeN, data = d1,
           corr = corCompSymm(form= ~ 1 | P.No) )
summary(fit)
intervals(fit)
# Coefficients:
#                Value     Std.Error  t-value    p-value
# (Intercept)    0.3549162 0.1736883  2.0434089  0.0463
# GazeNXSkCondN  0.4663221 0.3645852  1.2790484  0.2068
# DilN           0.3304675 0.1569353  2.1057567  0.0403
# GazeN         -0.0177445 0.3544598 -0.0500606  0.9603
# Degrees of freedom: 54 total; 50 residual
# Coefficients:
#                lower        est.       upper
# (Intercept)    0.006053003  0.35491622 0.7037794
# GazeNXSkCondN -0.265968761  0.46632207 1.1986129
# DilN           0.015253737  0.33046750 0.6456813
# GazeN         -0.729697972 -0.01774447 0.6942090

# Effects of sign(Fixations)Conductivity and Pupil on Affect
d1 = d
d1 <- groupedData(ES ~ GazeNXSkCondN + DilN
                  | P.No, data = d1)
fit <- gls(ES ~ GazeNXSkCondN + DilN, data = d1,
           corr = corCompSymm(form= ~ 1 | P.No) )
summary(fit)
intervals(fit)
# Coefficients:
#               Value     Std.Error t-value   p-value
# (Intercept)   0.3557055 0.1707441 2.083267  0.0423
# GazeNXSkCondN 0.4491561 0.1590911 2.823264  0.0068
# DilN          0.3292037 0.1533733 2.146422  0.0366
# Degrees of freedom: 54 total; 51 residual
# Coefficients:
#               lower      est.      upper
# (Intercept)   0.01292243 0.3557055 0.6984886
# GazeNXSkCondN 0.12976740 0.4491561 0.7685447
# DilN          0.02129407 0.3292037 0.6371134

# calculate effect size R^2 of linear model based on rmcorr
r = rmcorr(P.No, scale(0.4491561*GazeNXSkCondN + 0.3292037*DilN), 
           ES, d)  # r=0.567, R^2=0.322
r
plot(r, xlab="Predictor")


# Effects of sign(Blinks)Conductivity, Pupil and Blinks on Affect
d$BpmXSkCondN = sign(-scale(d$Bpm)) * d$SkCondN
d1 = d
d1 <- groupedData(ES ~ BpmXSkCondN + DilN + Bpm
                  | P.No, data = d1)
fit <- gls(ES ~ BpmXSkCondN + DilN + Bpm, data = d1,
           corr = corCompSymm(form= ~ 1 | P.No) )
summary(fit)
intervals(fit)
# Coefficients:
#             Value      Std.Error t-value     p-value
# (Intercept) -0.2295384 0.3482857 -0.6590522  0.5129
# BpmXSkCondN  0.5477556 0.2639888  2.0749194  0.0432
# DilN         0.3792769 0.1621254  2.3394055  0.0233
# Bpm          0.0050918 0.0027333  1.8628694  0.0684
# Degrees of freedom: 54 total; 50 residual
# Coefficients:
#             lower         est.         upper
# (Intercept) -0.9290907773 -0.229538429 0.47001392
# BpmXSkCondN  0.0175183863  0.547755583 1.07799278
# DilN         0.0536385940  0.379276945 0.70491530
# Bpm         -0.0003982156  0.005091807 0.01058183

# calculate effect size R^2 of linear model based on rmcorr
rmcorr(P.No, 0.5477556*BpmXSkCondN + 0.3792769*DilN 
       + 0.0050918*Bpm, ES, d) # r=0.497, R^2=0.247
