#Cleaning Prisoners Dilemma data for citalopram study
#Catherine Hobbs, University of Bath c.hobbs@bath.ac.uk
#24.03.2020
#RStudio Version 1.2.1335 Mac OS


#Libraries needed
library(tidyverse)
library(readxl)

prisoner_raw <- read_excel("/Volumes/files/RDSSD/Katie Studies/Citalopram/8. Data/Raw/Cognitive Tasks/Prisoner Dilemma/prisoner_raw_merged_anon.xlsx")

prisoner_clean <- prisoner_raw

#Note: below code not needed for my purposes as have removed this info when anonymising data (to prevent identification), however if you're using the task it may be useful to remove these columns to have a cleaner dataframe
#Drop Unwanted Variables
#prisoner_clean <- prisoner_clean[, -grep(".stopped$", colnames(prisoner_clean))]
#prisoner_clean <- prisoner_clean[, -grep(".started$", colnames(prisoner_clean))]
#prisoner_clean <- prisoner_clean[, -grep(".x$", colnames(prisoner_clean))]
#prisoner_clean <- prisoner_clean[, -grep(".y$", colnames(prisoner_clean))]
#prisoner_clean <- prisoner_clean[, -grep("Button$", colnames(prisoner_clean))]
#prisoner_clean <- prisoner_clean[, -grep("^Unnamed", colnames(prisoner_clean))]

#drops <- c("...1", "intro_mouse.clicked_name", "username_resp.keys", "name_mouse.clicked_name", "resp.clicked_name", "expName", "frameRate", "Unnamed: 438", "psychopyVersion", "date")
#prisoner_clean <- prisoner_clean[ , !(names(prisoner_clean) %in% drops)]
#rm(drops)

#Fill in values to whole column (e.g. avatar choice etc)
prisoner_clean <- prisoner_clean %>%
  group_by(participant) %>%
  fill(order) %>%
  fill(order, .direction = "down")

prisoner_clean <- prisoner_clean %>%
  group_by(participant) %>%
  fill(selected_avatar) %>%
  fill(selected_avatar, .direction = "down")

prisoner_clean <- prisoner_clean %>%
  group_by(participant) %>%
  fill(other_avatar) %>%
  fill(other_avatar, .direction = "down")

#Keeping only trial level rows
prisoner_clean <- subset(prisoner_clean, initial_choice_loop.thisRepN == 0 & trial_loop.thisRepN >= 0)

#Making social context a factor
prisoner_clean$initial_other_choice <- as.factor(prisoner_clean$initial_other_choice )

#Checking correct no. trials per participant & condition
check <- prisoner_clean %>%
  group_by(participant, initial_other_choice) %>%
  summarise(count=n())
rm(check)

#Multiplying resp.time by 1000 to show ms
prisoner_clean <- prisoner_clean %>%
  mutate(resp.time = resp.time*1000)

#Calculating proportion of cooperative choices
prisoner_clean$cooperation <- ifelse(prisoner_clean$choice == "Cooperate", 1, ifelse(prisoner_clean$choice == "Defect", 0, NA))

prisoner_clean <- prisoner_clean %>% 
  group_by(participant, initial_other_choice) %>% 
  mutate(cum_cooperation = cumsum(cooperation)) #Cumulative Cooperative Choices

prisoner_clean <- prisoner_clean %>% 
  group_by(participant, initial_other_choice) %>% 
  mutate(max_cooperation = max(cum_cooperation)) #Total Cooperative Choices

prisoner_clean$cooperation_proportion <- (prisoner_clean$max_cooperation / 26) * 100

#Removing unwanted columns
drops <- c("tl_p", "tl_o", "bl_p", "bl_o", "tr_p", "tr_o", "br_p", "br_o", "practice_intro_loop.thisRepN", "practice_intro_loop.thisTrialN", "practice_intro_loop.thisN", "initial_choice_loop.thisRepN", "initial_choice_loop.thisTrialN", "trial_loop.thisRepN", "trial_loop.thisTrialN", "left_text", "right_text")
prisoner_clean <- prisoner_clean[ , !(names(prisoner_clean) %in% drops)]
rm(drops)

#Calculating average RTs where people cooperated vs. defected
prisoner_clean_cooperate_rt <- prisoner_clean %>%
  filter(choice == "Cooperate") %>%
  group_by(participant, initial_other_choice) %>%
  select(participant, initial_other_choice, choice, resp.time) %>%
  summarise(resp.time_coop = mean(resp.time)) 

prisoner_clean_defect_rt <- prisoner_clean %>%
  filter(choice == "Defect") %>%
  group_by(participant, initial_other_choice) %>%
  select(participant, initial_other_choice, choice, resp.time) %>%
  summarise(resp.time_defect =mean(resp.time))

prisoner_clean_rt_choice <- merge(prisoner_clean_cooperate_rt, prisoner_clean_defect_rt, by = c("participant", "initial_other_choice"), all = TRUE)
rm(prisoner_clean_cooperate_rt, prisoner_clean_defect_rt)

#Trying to calculate the conditional probability of cooperating according to the proportion of rounds on which participants cooperated following each of the four possible outcomes.

#Creating a variable to show actions of both players per trial
prisoner_clean$trial_type <- ifelse(prisoner_clean$choice == "Cooperate" & prisoner_clean$other_choice == "Cooperate", 1, ifelse(prisoner_clean$choice == "Cooperate" & prisoner_clean$other_choice == "Defect", 2, ifelse(prisoner_clean$choice == "Defect" & prisoner_clean$other_choice == "Cooperate", 3, ifelse(prisoner_clean$choice == "Defect" & prisoner_clean$other_choice == "Defect", 4, NA))))

prisoner_clean$trial_type <- factor(prisoner_clean$trial_type,
                                    levels = c(1,2, 3, 4),
                                    labels = c("Both Cooperate", "Participant Cooperates, Other Defects", "Participant Defects, Other Cooperates", "Both Defect"))

#Lagging the action of both player to show previous trial
prisoner_clean <- prisoner_clean %>%
  group_by(participant, initial_other_choice) %>%
  mutate(previous_trial_type = lag(trial_type)) 

#Creating variables to show whether the action of players on the previous trial (separate variables for each type)
prisoner_clean$previous_both_cooperate <- ifelse(prisoner_clean$previous_trial_type == "Both Cooperate", 1, 0)
prisoner_clean$previous_both_cooperate[is.na(prisoner_clean$previous_both_cooperate)] = 0 #Entering as 0 so can total (as 1 indicates yes which is what we're trying to establish)

prisoner_clean$previous_p_coop_o_defect <- ifelse(prisoner_clean$previous_trial_type == "Participant Cooperates, Other Defects", 1, 0)
prisoner_clean$previous_p_coop_o_defect[is.na(prisoner_clean$previous_p_coop_o_defect)] = 0 #Entering as 0 so can total (as 1 indicates yes which is what we're trying to establish)

prisoner_clean$previous_p_defect_o_coop <- ifelse(prisoner_clean$previous_trial_type == "Participant Defects, Other Cooperates", 1, 0)
prisoner_clean$previous_p_defect_o_coop[is.na(prisoner_clean$previous_p_defect_o_coop)] = 0 #Entering as 0 so can total (as 1 indicates yes which is what we're trying to establish)

prisoner_clean$previous_both_defect <- ifelse(prisoner_clean$previous_trial_type == "Both Defect", 1, 0)
prisoner_clean$previous_both_defect[is.na(prisoner_clean$previous_both_defect)] = 0 #Entering as 0 so can total (as 1 indicates yes which is what we're trying to establish)


#Totalling no. trials
prisoner_clean <- prisoner_clean %>% 
  group_by(participant, initial_other_choice) %>% 
  mutate(sum_previous_both_cooperate = sum(previous_both_cooperate), sum_previous_p_coop_o_defect = sum(previous_p_coop_o_defect), sum_previous_p_defect_o_coop = sum(previous_p_defect_o_coop), sum_previous_both_defect = sum(previous_both_defect))

#Creating variables to show whether the person cooperated depending upon the action of players on the previous trial (separate variables for each type)
prisoner_clean$previous_both_cooperate_choice <- ifelse(prisoner_clean$previous_both_cooperate == 1 & prisoner_clean$cooperation == 1, 1, ifelse(prisoner_clean$previous_both_cooperate == 1 & prisoner_clean$cooperation == 0, 0, NA))
prisoner_clean$previous_both_cooperate_choice[is.na(prisoner_clean$previous_both_cooperate_choice)] = 0 #Entering as 0 so can total

prisoner_clean$previous_p_coop_o_defect_choice <- ifelse(prisoner_clean$previous_p_coop_o_defect == 1 & prisoner_clean$cooperation == 1, 1, ifelse(prisoner_clean$previous_p_coop_o_defect == 1 & prisoner_clean$cooperation == 0, 0, NA))
prisoner_clean$previous_p_coop_o_defect_choice[is.na(prisoner_clean$previous_p_coop_o_defect_choice)] = 0 #Entering as 0 so can total

prisoner_clean$previous_p_defect_o_coop_choice <- ifelse(prisoner_clean$previous_p_defect_o_coop == 1 & prisoner_clean$cooperation == 1, 1, ifelse(prisoner_clean$previous_p_defect_o_coop == 1 & prisoner_clean$cooperation == 0, 0, NA))
prisoner_clean$previous_p_defect_o_coop_choice[is.na(prisoner_clean$previous_p_defect_o_coop_choice)] = 0 #Entering as 0 so can total

prisoner_clean$previous_both_defect_choice <- ifelse(prisoner_clean$previous_both_defect == 1 & prisoner_clean$cooperation == 1, 1, ifelse(prisoner_clean$previous_both_defect == 1 & prisoner_clean$cooperation == 0, 0, NA))
prisoner_clean$previous_both_defect_choice[is.na(prisoner_clean$previous_both_defect_choice)] = 0 #Entering as 0 so can total

#Counting no. trials where they cooperated
prisoner_clean <- prisoner_clean %>% 
  group_by(participant, initial_other_choice) %>% 
  mutate(sum_previous_both_cooperate_choice = sum(previous_both_cooperate_choice), sum_previous_p_coop_o_defect_choice = sum(previous_p_coop_o_defect_choice), sum_previous_p_defect_o_coop_choice = sum(previous_p_defect_o_coop_choice), sum_previous_both_defect_choice = sum(previous_both_defect_choice))

#Establishing proportion of trials cooperated according to trial type
prisoner_clean <- prisoner_clean %>% 
  group_by(participant, initial_other_choice) %>% 
  mutate(prop_previous_both_cooperate_choice = ((sum_previous_both_cooperate_choice/sum_previous_both_cooperate)*100), prop_previous_p_coop_o_defect_choice = ((sum_previous_p_coop_o_defect_choice/sum_previous_p_coop_o_defect)*100), prop_previous_p_defect_o_coop_choice = ((sum_previous_p_defect_o_coop_choice/sum_previous_p_defect_o_coop)*100), prop_previous_both_defect_choice = ((sum_previous_both_defect_choice/sum_previous_both_defect)*100))

#Creating aggregated dataset
prisoner_clean_aggregate <- prisoner_clean %>%
  group_by(participant, initial_other_choice) %>%
  select(participant, initial_other_choice, other_points_total, participant_points_total, resp.time, cooperation_proportion, max_cooperation, prop_previous_both_cooperate_choice, prop_previous_p_coop_o_defect_choice, prop_previous_p_defect_o_coop_choice, prop_previous_both_defect_choice) %>%
  summarise_all(funs(mean(.))) %>%
  rename(cooperation_total = max_cooperation)

#Replacing NaN with 0 (note: this happened where 0 was divided by 0 e.g. when there were no trials where participants cooperated etc)
prisoner_clean_aggregate$prop_previous_both_cooperate_choice[is.nan(prisoner_clean_aggregate$prop_previous_both_cooperate_choice)]<-0
prisoner_clean_aggregate$prop_previous_p_coop_o_defect_choice[is.nan(prisoner_clean_aggregate$prop_previous_p_coop_o_defect_choice)]<-0
prisoner_clean_aggregate$prop_previous_p_defect_o_coop_choice[is.nan(prisoner_clean_aggregate$prop_previous_p_defect_o_coop_choice)]<-0
prisoner_clean_aggregate$prop_previous_both_defect_choice[is.nan(prisoner_clean_aggregate$prop_previous_both_defect_choice)]<-0

#Merging with RT according to cooperative or defective choices
prisoner_clean_aggregate <- merge(prisoner_clean_aggregate, prisoner_clean_rt_choice, by = c("participant", "initial_other_choice"))

#Merging with drug matching file
blind_drug_matching <- read_excel("/Volumes/files/RDSSD/Katie Studies/Citalopram/8. Data/Raw/Group Allocation/Blinded_drug_matching_KH_format_for_R_anon.xlsx")
prisoner_clean_aggregate <- merge(blind_drug_matching, prisoner_clean_aggregate, by = "participant")

#Saving file
save(prisoner_clean_aggregate, file = "/Volumes/files/RDSSD/Katie Studies/Citalopram/8. Data/Analysis/Prisoner/prisoner_clean_aggregate")
write.csv(prisoner_clean_aggregate, "/Volumes/files/RDSSD/Katie Studies/Citalopram/8. Data/Analysis/Prisoner/prisoner_clean_aggregate.csv")






